/*      Copyright (c) 1996 Mix Software, Inc.

        w32io.c - module to read and write blocks from an open file
                (internal use only)

        The file is assumed to be a collection of blksize-sized blocks.
        The buffer must be large enough to hold blksize bytes.
*/
#include <string.h>
#include "btree.h"
#include "blockio.h"
/*  eliminate unused windows features for faster compiles */
#define NOGDICAPMASKS
#define NOVIRTUALKEYCODES
#define NOWINMESSAGES
#define NOWINSTYLES
#define NOSYSMETRICS
#define NOMENUS
#define NOICONS
#define NOKEYSTATES
#define NOSYSCOMMANDS
#define NORASTEROPS
#define NOSHOWWINDOW
#define OEMRESOURCE
#define NOATOM
#define NOCLIPBOARD
#define NOCOLOR
#define NODRAWTEXT
#define NONLS
#define NOMEMMGR
#define NOMETAFILE
#define NOMINMAX
#define NOOPENFILE
#define NOSCROLL
#define NOSERVICE
#define NOSOUND
#define NOTEXTMETRIC
#define NOWH
#define NOWINOFFSETS
#define NOCOMM
#define NOKANJI
#define NOHELP
#define NOPROFILER
#define NODEFERWINDOWPOS
#include <windows.h>
#include <mmsystem.h>

#if !defined WRITE_THROUGH
#define WRITE_THROUGH 0
#endif
#if WRITE_THROUGH
#define FILE_FLAGS FILE_FLAG_RANDOM_ACCESS | FILE_FLAG_WRITE_THROUGH
#else
#define FILE_FLAGS FILE_FLAG_RANDOM_ACCESS
#endif

#define USE_STD_IO    1
#if defined(__DLL__) || defined(I_DLL)
#define STDOUT_HANDLE 1
#endif
#if USE_STD_IO
#include <stdio.h>
#endif

int _cbt_doserrno = 0;
int _b_no_flush = 0;

/*
        Defined functions (CBT internal use only)

        cb_openfile               open a random file
        cb_close                  close a file
        cb_read                   read a random area from a file
        cb_readblk                read block from file
        cb_readseq                read sequentially
        cb_write                  write to a file
        cb_writeblk               write block to file
        cb_writeseq               write sequentially
        cb_commitblk              commit updates to disk
        cb_lock_area              lock a portion of a file
        cb_unlock_area            unlock a portion of a file
        cb_delay                  delay (before lock retries)
        cb_prints                 print a string to stdout
        cb_can_share              can files be shared?
*/
int cb_read(int fd, unsigned long location, int size, void *buffer);
int cb_write(int fd, unsigned long location, int size, void *buffer);

/**/
/* ------------------------------------------------------------------------
        cb_openfile - open a block oriented file

        Filename is the name of the file to be opened.  Share is either I_YES
        or I_NO, (1 or 0).  If share is I_YES, the file is opened for shared
        access so that multiple programs can access the file at the same time.

        Return a non-negative integer on success or -1 on error.  In the
        event of an error, errno is also set.
*/

int cb_openfile(char *filename, int mode, int share)
{
   HANDLE fd;
   ULONG attrib;
   DWORD modeflg = 0;

   if (mode == CBT_OPEN_RDONLY) attrib = GENERIC_READ;
   else attrib = GENERIC_READ | GENERIC_WRITE;
   if (share) modeflg = FILE_SHARE_READ | FILE_SHARE_WRITE;

   fd = CreateFile(filename, attrib, modeflg, NULL, OPEN_EXISTING,
            FILE_FLAGS, NULL);
   if (fd == INVALID_HANDLE_VALUE) {
      _cbt_doserrno = GetLastError();
      return -1;
      }
   return (int) fd;
} /* cbopenfile */
/**/
/* ------------------------------------------------------------------------
        cb_create - create a file and open it for exclusive write

        Filename is the name of the file to be opened.
*/

int cb_create(char *filename)
{
   HANDLE fd;

   fd = CreateFile(filename, GENERIC_READ | GENERIC_WRITE, 0, NULL, CREATE_NEW,
            FILE_FLAGS, NULL);
   if (fd == INVALID_HANDLE_VALUE) {
      _cbt_doserrno = GetLastError();
      return -1;
      }
   return (int) fd;
} /* cb_create */
/**/
/* ------------------------------------------------------------------------

        cbcloseblk - close a block oriented file

        Return 0 on success or -1 on error.
*/

int cb_closeblk(int fd)
{
   if (CloseHandle((HANDLE) fd) != TRUE) {
      _cbt_doserrno = GetLastError();
      return -1;
      }
   return 0;
} /* cb_closeblk */
/**/
/* ------------------------------------------------------------------------

        cbreadblk - read one block of blksize from file into buffer

        Return OK or ERROR.
*/
int cb_readblk(int fd, Blk_Nbr blocknbr, int blksize, void *buffer)
{
   return cb_read(fd, blocknbr * blksize, blksize, buffer);
   }

/* ------------------------------------------------------------------------

        cb_read  - random read

        Return OK or ERROR.
*/

int cb_read(int fd, unsigned long location, int size, void *buffer)
{
   DWORD  readlength;
   DWORD  newlocation;

   /* seek to position */
   newlocation = SetFilePointer((HANDLE) fd, location, NULL, FILE_BEGIN);
   if (newlocation != location) {
      _cbt_doserrno = GetLastError();
      return I_ERROR;
      }

   /* read data from file */
   if (ReadFile((HANDLE) fd, buffer, size, &readlength, NULL) != TRUE) {
      _cbt_doserrno = GetLastError();
      return I_ERROR;
      }
   if (readlength < (DWORD) size) {
      _cbt_doserrno = EC_DOS_READERR;
      return I_ERROR;
      }
   return I_OK;
   } /* cbreadblk */

/* ------------------------------------------------------------------------

        cb_readseq  - sequential read

        Return OK or ERROR.
*/

int cb_readseq(int fd, int size, void *buffer)
{
   ULONG  readlength;

   if (ReadFile((HANDLE) fd, buffer, size, &readlength, NULL) != TRUE) {
      _cbt_doserrno = GetLastError();
      return I_ERROR;
      }
   if (readlength < (DWORD) size) {
      _cbt_doserrno = EC_DOS_READERR;
      return I_ERROR;
      }
   return I_OK;
   } /* cbreadseq */
/**/
/* ------------------------------------------------------------------------

        cbwriteblk - write one block of blksize to file from buffer

        Return OK or ERROR.
*/

int cb_writeblk(int fd, Blk_Nbr blocknbr, int blksize, void *buffer)
{
   return cb_write(fd, blocknbr * blksize, blksize, buffer);
   }

/* ------------------------------------------------------------------------

        cb_write - random write

        Return OK or ERROR.
*/

int cb_write(int fd, unsigned long location, int size, void *buffer)
{
   DWORD  count;
   DWORD  newlocation;

   /* seek to position */
   newlocation = SetFilePointer((HANDLE) fd, location, NULL, FILE_BEGIN);
   if (newlocation != location) {
      _cbt_doserrno = GetLastError();
      return I_ERROR;
      }

   /* write data to file */
   if (WriteFile((HANDLE) fd, buffer, size, &count, NULL) != TRUE) {
      _cbt_doserrno = GetLastError();
      return I_ERROR;
      }
   if (count < (DWORD) size) {
      _cbt_doserrno = EC_DOS_READERR;
      return I_ERROR;
      }
   return I_OK;
   } /* cb_write */

/* ------------------------------------------------------------------------

        cb_writeseq - sequential write

        Return OK or ERROR.
*/

int cb_writeseq(int fd, int size, void *buffer)
{
   DWORD  count;

   if (WriteFile((HANDLE) fd, buffer, size, &count, NULL) != TRUE) {
      _cbt_doserrno = GetLastError();
      return I_ERROR;
      }
   if (count < (DWORD) size) {
      _cbt_doserrno = EC_DOS_READERR;
      return I_ERROR;
      }
   return I_OK;
   } /* cb_writeseq */

/**/
/* ------------------------------------------------------------------------

        cb_remove - delete a file

        Return OK or ERROR.
*/

int cb_remove(char *filename)
{
   if (DeleteFile(filename) != TRUE) {
      _cbt_doserrno = GetLastError();
      return I_ERROR;
      }
   return I_OK;
   } /* cb_remove */

/**/
/* ------------------------------------------------------------------------

        cb_lock_area - put a record lock on a file region

        Return OK or ERROR.
*/

int cb_lock_area(int fd, unsigned long location, long size, int tries,
   long delay, long timeout)
{
   DWORD start_time;

   if (LockFile((HANDLE) fd, location, 0, size, 0) == TRUE) return I_OK;
   start_time = timeGetTime();

   while (--tries > 0) {
      Sleep(delay);
      if (LockFile((HANDLE) fd, location, 0, size, 0) == TRUE) return I_OK;
      if ((long)(timeGetTime() - start_time) > timeout) return I_ERROR;
      }
   return I_ERROR;
} /* cb_lock_area */
/**/
/* ------------------------------------------------------------------------

        cb_unlock_area - remove a record lock from a file region

        Return OK or ERROR.
*/

int cb_unlock_area(int fd, unsigned long location, long size)
{
   if (UnlockFile((HANDLE) fd, location, 0, size, 0) != TRUE) {
      _cbt_doserrno = GetLastError();
      return I_ERROR;
      }
   return I_OK;
} /* cb_unlock_area */
/**/
/* ------------------------------------------------------------------------

        cb_commitblk -   flush any system buffer associated with specified
                         file descriptor

        Return OK or ERROR.
*/

int cb_commitblk(int fd)
{
   if (_b_no_flush) return I_OK;
   if (FlushFileBuffers((HANDLE) fd) != TRUE) {
      _cbt_doserrno = GetLastError();
      return I_ERROR;
      }
   return I_OK;
   }
/**/
/* ------------------------------------------------------------------------

        cb_delay - time delay before lock retry

        always returns OK
*/

int cb_delay(long time)
{
   Sleep(time);
   return I_OK;
} /* cb_delay */

/**/
/* ------------------------------------------------------------------------

        cb_prints - print a string to stdout

*/
#if USE_STD_IO
   #if defined(I_DLL) || defined(__DLL__)

   int I_ENTRY cb_prints(const char *string)
   {
      static first_time = 1;
      if (first_time) {
         setbuf(stdout, NULL);
         first_time = 0;
         }
      if (fputs(string, stdout) != 0)
         return I_ERROR;
      return I_OK;
      } /* cb_prints */

   #else

   int I_ENTRY cb_prints(const char *string)
   {
      if (fputs(string, stdout) != 0)
         return I_ERROR;
      return I_OK;
      } /* cb_prints */
   #endif
#else
int I_ENTRY cb_prints(const char *string)
{
   MessageBox(NULL, string, "C/Database Toolchest",
              MB_ICONINFORMATION | MB_OK | MB_TASKMODAL);
   return I_OK;
} /* cb_prints */
#endif
/**/

/* ------------------------------------------------------------------------

        cb_can_share - can files be shared?

        Return I_YES or I_NO.
*/

int cb_can_share(void)
{
   return I_YES;
} /* cb_can_share */
/**/

/* ------------------------------------------------------------------------

        cb_filesize - determine the size of an open file

        Return OK or ERROR.
*/

int cb_filesize(int fd, long *size)
{
   DWORD err;
   *size = GetFileSize((HANDLE) fd, NULL);
   if ((DWORD)(*size) == 0xFFFFFFFF) {
      err = GetLastError();
      if (err != NO_ERROR) {
         _cbt_doserrno = err;
         return I_ERROR;
         }
      }
   return (I_OK);
   } /* cb_filesize */

/* ------------------------------------------------------------------------

        cbset_print - set the print function
        Return OK or ERROR.
*/

I_EXPORT Puts_Fn_Ptr I_ENTRY cbset_print(Puts_Fn_Ptr printfn)
{
   Puts_Fn_Ptr old;
   old = _cb_prints;
   _cb_prints = printfn;
   return old;
   } /* cbset_print */

/* end w32io.c */
