/*      Copyright (c) 1995 Mix Software, Inc.

        path.c - routines handling paths and drive specifiers for filenames

        All of these routines assume a syntactically correct filename,
        and they may not recover gracefully given an ill-formed filename.

        These routines are operating system dependent.
*/
#include "cdtdef.h"
#include "filename.h"
#include <string.h>
#if defined(__TURBOC__)
   #include <dir.h>
#elif defined(__EMX__)
   #include <stdlib.h>
#else
   #include <direct.h>
#endif

/*
        Defined Functions

        abs_filename            construct absolute filename
        pick_drive              extract drive letter from filename
        expanddrive             put drive name in buffer
        pick_path               extract path from filename
        expandpath              put path in buffer
        pick_name               extract name from filename
        pick_extent             extract extent from filename
*/
/**/

#if !defined(__POWERC) && !defined(__TURBOC__) && !defined(__BORLANDC__)
extern int getcurdir(int drive, char *buf);
extern int getdisk(void);
#endif

/*
   abs_filename - convert filename into absolute filename with a
         fully-qualified name including drive specifier and path.

   If the absolute filename won't fit in the buffer, it is an error.

   Returns pointer to absolute filename, or NULL for error.
   */
/**/


char *abs_filename(char *filename, char *buffer, int bufsize)
{
   char *bufp;
   int  length;
   char pickbuf[MAX_FILENAME + 1];
   char drive[_MAX_DRIVE + 1];
   char path[MAX_FILENAME + 1];
   char name[MAX_FILENAME + 1];
   char extent[MAX_FILENAME + 1];

   if (D2_SEP != DIR_SEP) {
      for (bufp = filename; *bufp; bufp++) {
         if (*bufp == D2_SEP) *bufp = DIR_SEP;
         }
      }
   bufp = buffer;
   if (pick_drive(filename, pickbuf, MAX_FILENAME + 1) == I_ERROR)
      return (NULL);
   length = expanddrive(pickbuf, drive, sizeof(drive));
   if (length == I_ERROR || length + 1 > bufsize) return (NULL);
   strcpy(bufp, drive);
   bufp += length;
   bufsize -= length;
   if (pick_path(filename, pickbuf, sizeof(pickbuf)) == I_ERROR) return (NULL);
   length = expandpath(drive, pickbuf, path, sizeof(path));
   if (length == I_ERROR || length + 1 > bufsize) return (NULL);
   strcpy(bufp, path);
   bufp += length;
   bufsize -= length;
   length = pick_name(filename, name, sizeof(name));
   if (length == I_ERROR || length + 1 > bufsize) return (NULL);
   strcpy(bufp, name);
   bufp += length;
   bufsize -= length;
   length = pick_extent(filename, extent, sizeof(extent));
   if (length == I_ERROR || length + 1 > bufsize) return (NULL);
   strcpy(bufp, extent);
   return (buffer);
   } /* abs_filename */
/**/
/*
        pick_drive - pick drive specifier from filename and put in buffer

        Returns length of drive string (0 if no drive)
        and fills in buffer (inserts a nul if no drive),
        or I_ERROR for syntax error in input filename
*/

int pick_drive(char *filename, char *buffer, int bufsize)
{
   char *bufp;
   char *endp;
   int length;

   bufp = buffer;
   if ((endp = strchr(filename, DRV_SEP)) != NULL) {
      length = (int) (endp - filename + 1);
      if (length != _MAX_DRIVE-1 || length + 1 > bufsize) return (I_ERROR);
      while (length-- > 0) *bufp++ = *filename++;
      }
   else if (*filename == DIR_SEP && *(filename+1) == DIR_SEP) {
      *bufp++ = DIR_SEP;   /* allow network name 12/12/96 */
      }
   else if (bufsize < 1) return (I_ERROR);
   *bufp = '\0';
   strupr(buffer);
   return (bufp - buffer);
   } /* pick_drive */
/**/
/*
        expanddrive - expand drivename into buffer

        If drivename is a null string, calls uses current drive.

        Returns length of expanded drivename and fills in buffer,
        or I_ERROR
*/

int expanddrive(char *drivename, char *buffer, int bufsize)
{
   char *bufp;
   int drivenbr;
   int length;

   if (bufsize < _MAX_DRIVE) return (I_ERROR);
   if (*drivename == DIR_SEP) { /* network name */
      *buffer = '\0';
      return 0;
      }
   if (*drivename) {
      length = strlen(drivename);
      if (length + 1 > bufsize) return (I_ERROR);
      strcpy(buffer, drivename);
      return (length);
      }
   else {
      bufp = buffer;
      drivenbr = getdisk();
      *bufp++ = (char) (drivenbr + 'A');
      *bufp++ = DRV_SEP;
      *bufp = '\0';
      return (bufp - buffer);
      }
   } /* expanddrive */
/**/
/*
        pick_path - pick pathname from filename and put in buffer

        Return length of pathname string (0 if no path)
        and fills in buffer (insert '\0' if no path),
        or I_ERROR
*/

int pick_path(char *filename, char *buffer, int bufsize)
{
   char *bufp;
   char *tempp;
   char *endp;          /* points to end of pathname */
   int length;

   bufp = buffer;
   if ((tempp = strchr(filename, DRV_SEP)) != NULL) filename = ++tempp;
   if ((endp = strrchr(filename, DIR_SEP)) != NULL) {
      endp++;         /* include the slash */
      if (strcmp(endp, LAST_DIR) == 0) endp += 2;
      else if (strcmp(endp, THIS_DIR) == 0) endp++;
      length = (int) (endp - filename + 1);
      if ((length + 1 > bufsize) || (length > _MAX_PATH)) return (I_ERROR);
      while (filename < endp) {
         *bufp++ = *filename++;
         }
      }
   else if (bufsize < 1) return (I_ERROR);
   *bufp = '\0';
   strupr(buffer);
   return (bufp - buffer);
   } /* pick_path */
/**/
/*
        expandpath - expand pathname for drive into buffer

        Uses current path on current drive if no pathname given.

        Returns length of expanded pathname and fills in buffer,
        or I_ERROR
*/

int expandpath(char *drive, char *pathname, char *buffer, int bufsize)
{
   char curdir[MAX_FILENAME + 1 + 1 + 1];  /* current directory */
   /* includes room for leading/trailing slashes and nul */
   int curdirlen;              /* length of current directory */
   char *endp;                 /* pts to last char of curdir */
   int drivenbr;               /* nbr of current drive */
   int pathlen;                /* length of pathname */

   pathlen = strlen(pathname);
   if (*pathname == DIR_SEP) {
      if (pathlen + 1 > bufsize) return (I_ERROR);
      strcpy(buffer, pathname);
      return (pathlen);
      }
   *curdir = DIR_SEP;
   if (*drive) drivenbr = *drive - 'A' + 1;
   else drivenbr = 0;
   if (getcurdir(drivenbr, curdir + 1) == -1) return (I_ERROR);
   endp = curdir + strlen(curdir) - 1;
   if (*endp != DIR_SEP) *++endp = DIR_SEP;
   for (;;) {
      if (strncmp(pathname, THIS_DIR_SEP, 2) == 0) {
         pathname += 2;
         pathlen -= 2;
         }
      else if (strncmp(pathname, LAST_DIR_SEP, 3) == 0) {
         pathname += 3;
         pathlen -= 3;
         if (--endp < curdir) return (I_ERROR);
         while (*endp != DIR_SEP) endp--;
         }
      else break;
      } /* for */
   curdirlen = (int) (endp - curdir + 1);
   if (curdirlen + pathlen + 1 > bufsize) return (I_ERROR);
   strncpy(buffer, curdir, curdirlen);
   strcpy(buffer + curdirlen, pathname);
   return (curdirlen + pathlen);
   } /* expandpath */
/**/
/*
        pick_name - pick name part from filename and put in buffer

        Return length of name part (0 if no name)
        and fill in buffer (inserts '\0' if no name part),
        or I_ERROR
*/

int pick_name(char *filename, char *buffer, int bufsize)
{
   char *tempp;
   int length;

   if (bufsize < 1) return (I_ERROR);
   if ((tempp = strchr(filename, DRV_SEP)) != NULL) filename = ++tempp;
   if ((tempp = strrchr(filename, DIR_SEP)) != NULL) filename = ++tempp;
   if (strcmp(filename, LAST_DIR) == 0 || strcmp(filename, THIS_DIR) == 0) {
      *buffer = '\0';
      return (0);
      }
   if ((tempp = strrchr(filename, EXT_SEP)) != NULL)
      length = (int) (tempp - filename);
   else length = strlen(filename);
   if (length > _MAX_FNAME || length + 1 > bufsize) return (I_ERROR);
   if (length) strncpy(buffer, filename, length);
   buffer [length] = '\0';
   strupr(buffer);
   return (length);
   } /* pick_name */
/**/
/*
        pick_extent - pick extent from filename and put in buffer

        Extent includes the '.' as the first character

        Return length of extent part (0 if no extent)
        and fills in buffer (inserts '\0' if no extent),
        or I_ERROR
*/

int pick_extent(char *filename, char *buffer, int bufsize)
{
   char *tempp;
   int length;

   if (bufsize < 1) return (I_ERROR);
   *buffer = '\0';
   length = 0;
   for (tempp = filename + strlen(filename) - 1; tempp > filename; tempp--) {
      if (*tempp == DIR_SEP || *tempp == DRV_SEP) return (0);
      if (*tempp == EXT_SEP) {
         length = strlen(tempp);
         if (length == 1) return (0);
         if (length > _MAX_EXT || length + 1 > bufsize) return (I_ERROR);
         strcpy(buffer, tempp);
         break;
         }
      }
   strupr(buffer);
   return (length);
   } /* pick_extent */

/* end path.c */
