/*      Copyright (c) 1987 Shrier and Deihl. Licensed to Mix Software, Inc.
        Copyright (c) 1995 Mix Software, Inc. Multi-user version

        opendb.c - ISAM iopen_db and supporting functions
*/
#include "isamlib.h"
#include "isamerr.h"
#include "filename.h"
#include "blockio.h"
#include "cberr.h"
#include <stdlib.h>
#include <string.h>
#ifndef min
#define min(a,b)        ((a) < (b) ? (a) : (b))
#endif

/*
        Defined Functions

        iopen_db                open an existing database
        iopen_db_shared         share an existing database
        I_ifopen                INTERNAL open the index file
        I_compare               key compare routine
*/
int I_ENTRY I_compare(char *keya, int lena, char *keyb, int lenb);

/* open a single index */
Index_Obj *I_iopen(Db_Obj *db, char *index_desc, int desc_length, int nbr,
                   int shared, int readonly);

/* alloc and init a Db_Obj */
static Db_Obj *I_new_db_obj(void);

/* open the data file */
static int I_dfopen(Db_Obj *db, char *data_file, int share, int readonly);

/* read header from data file */
static int I_rd_header(Db_Obj *db);

/* fake a physical index */
static int I_kludge_physical(Db_Obj *db);

/* alloc and init an Index_Obj */
static Index_Obj *I_new_index_obj(void);

/* find next index in index file */
static int I_next_index(void *tree, char *index_buf, int index_buf_len,
                        int *desc_length, int *index_nbr);

/* extract description of index */
static int I_pull_desc(Db_Obj *db, char *buf, int buflen, Index_Obj *index);

/* count fields in buffer */
static int I_count_fields(char *buffer, int buflen);

/* find name in buffer of names */
static int I_mat_namelist(char *buffer, char *names[], int nbr_of_names);

/**/
/*
        iopen_db - open an existing database (data and index file)

        If the index file cannot be opened, or it is garbled, the database is
        still opened without error, but only the Physical Index will be
        available.  You can remake a garbled index file with inew_index_file.
        If the index file is not opened, iopen_db still returns a valid
        database handle, but isam_errno is set to the error (that is, it is
        not I_NOERR).

        It is an error if the data file for the database cannot be opened
        (but not if the index file cannot be opened). It is an error if the
        database exists and is already open.

        Returns database handle, or NULL for error.
*/

I_EXPORT Db_Obj * I_ENTRY iopen_db(const char *db_name)
{
   return iopen_db_shared(db_name, I_NO, I_NO);
   }

/*
        iopen_db_shared - open a shared database (data and index file)
*/
I_EXPORT Db_Obj * I_ENTRY iopen_db_shared(const char *db_name, int share,
              int readonly)
{
   Db_Obj *db;
   char dataname[MAX_FILENAME + 1];
   char indexname[MAX_FILENAME + 1];

   SET_EC(I_NOERR)
   if (cbbufsize() == 0) {
      if (isaminit(DEFAULT_BUFCNT, DEFAULT_BUFSIZE) != I_OK)
         return ((Db_Obj *) NULL);
      }
   if (idbhandle(db_name) != NULL) {
      SET_EC(I_INUSE)
      return ((Db_Obj *) NULL);
      }
   if (I_db_key.key_buf == NULL) {
      I_db_key.key_buf_len = cbbufsize();
      if ((I_db_key.key_buf = (char *) malloc(I_db_key.key_buf_len)) == NULL) {
         SET_EC(I_OOM)
         return ((Db_Obj *) NULL);
         }
      I_db_key.key_len = 0;
      }

   if (I_mkdata_filename(db_name, dataname, MAX_FILENAME) == NULL
       || I_mkindex_filename(db_name, indexname, MAX_FILENAME) == NULL) {
      SET_EC(I_MAKENAME)
      return ((Db_Obj *) NULL);
      }
   if ((db = I_new_db_obj()) == NULL) {
      /* isam_errno already set */
      return ((Db_Obj *) NULL);
      }
   if (I_dfopen(db, dataname, share, readonly) == I_ERROR) {
      I_dfclose(db);
      /* isam_errno already set */
      return ((Db_Obj *) NULL);
      }
   /*
           open index file and all internal indexes
   */
   if (I_ifopen(db, indexname, share, readonly) == I_ERROR) {
      I_ifclose(db);
      if (cberrcode == EC_LOCK) { /* unable to read control record */
         SET_EC(I_LOCKERR)
         I_dfclose(db);           /* index exists but is not available */
         return ((Db_Obj *) NULL);  /* make open fail */
         }
      /* allow user to continue processing */
      if (I_kludge_physical(db) == I_ERROR) {
         I_dfclose(db);
         return ((Db_Obj *) NULL);
         }
      }
   db->share = share;
   db->readonly = readonly;
   if (share) db->locked = 0; else db->locked = I_READLOCK | I_WRITELOCK;
   return (db);
   } /* iopen_db */
/**/
/*
        I_iopen - open one index in the cbt file

        Returns pointer to index object, or NULL for error.
*/

Index_Obj *I_iopen(Db_Obj *db, char *index_desc, int desc_length, int nbr,
                   int shared, int readonly)
   /* db          = database to look at */
   /* index_desc  = description of index to open */
   /* desc_length = length of description */
   /* nbr         = index number assigned */
   /* shared      = I_YES if the database is shared with other tasks */
{
   Index_Obj *index;
   char *name;

   if ((index = I_new_index_obj()) == NULL) {
      /* isam_errno already set */
      return ((Index_Obj *) NULL);
      }
   name = index_desc + 1;
   if ((index->index_name = (char *) malloc(strlen(name) + 1)) == NULL) {
      free(index);
      SET_EC(I_OOM)
      return ((Index_Obj *) NULL);
      }
   strcpy(index->index_name, name);
   index->index_nbr = nbr;
   if (index->index_nbr == PHYSICAL_INDEX) index->tree = NULL;
   else {
      if ((index->tree =
           cbopen_shared(db->index_filename, I_compare, shared, readonly))
           == NULL) {
         free(index->index_name);
         free(index);
         SET_EC(I_CBT)
         return ((Index_Obj *) NULL);
         }
      }
   index->db = db;
   if (I_pull_desc(db, index_desc, desc_length, index) != I_OK) {
      free(index->index_name);
      free(index);
      /* isam_errno already set */
      return ((Index_Obj *) NULL);
      }
   return (index);
   } /* I_iopen */
/**/
/*
        I_new_db_obj - alloc space and initialize new Db_Obj

        Returns pointer to new Db_Obj, or NULL if out of memory.
*/

static Db_Obj *I_new_db_obj()
{
   Db_Obj *db;

   if ((db = (Db_Obj *) malloc(sizeof(Db_Obj))) == NULL) {
      SET_EC(I_OOM)
      return ((Db_Obj *) NULL);
      }
   I_db_object = db;
   db->data_filename = NULL;
   db->fd = -1;
   db->field_count = 0;
   db->field_names = NULL;
   db->field_buf = NULL;
   db->first_rec = 0;
   db->field_type = NULL;
   db->field_len = NULL;
   db->min_rec_size = sizeof(Rec_Len) + 1;
   db->readonly = I_NO;
   db->share = I_NO;
   db->locked = 0;
   db->index_filename = NULL;
   db->desc_tree = NULL;
   db->hole_tree = NULL;
   db->index_count = 0;
   db->index_list = NULL;
   db->locks.next = NULL;
   db->locks.location = NO_CURRENT;
   db->locks.handle = 1;
   db->liblock = NO_CURRENT;
   db->phys_ptr = NO_CURRENT;
   add_member((void **) &I_db_objs, (Any_Set *) db);
   return (db);
   } /* I_new_db_obj */
/**/
/*
        I_dfopen - open the data file and read its header into Db_Obj

        Returns I_OK or I_ERROR.
*/

static int I_dfopen(Db_Obj *db, char *data_file, int share, int readonly)
   /* db        = OUTPUT filled in with stuff from data file */
   /* data_file = name of data file to open */
   /* share     = I_YES to open the file shared */
   /* readonly  = I_YES to open the file with read only access */
{
   if ((db->data_filename = (char *) malloc(strlen(data_file) + 1)) == NULL)
      ERRX(I_OOM)
   strcpy(db->data_filename, data_file);

   if (readonly) db->fd = _cb_openfile(data_file, CBT_OPEN_RDONLY, share);
   else db->fd = _cb_openfile(data_file, CBT_OPEN_RDWR, share);
   if (db->fd == -1) ERRX(I_NOTFILE)
   db->share = share;
   db->readonly = readonly;
   if (I_rd_header(db) == I_ERROR) {
      /* isam_errno already set */
      return (I_ERROR);
      }
   return (I_OK);
   } /* I_dfopen */
/**/
/*
        I_rd_header - read header from data file and set up pointers to
        field names

        Returns I_OK or I_ERROR.
*/

static int I_rd_header(Db_Obj *db)
{
   Rec_Len names_length;
   int i;
   int len, lock;
   int unlock = 0;
   char *p;
   char *cptr;
   int type;
   int j;
   long min_rec_size, length;

   lock = sizeof(names_length) + sizeof(db->field_count);
   if (db->share) {
      if (_cb_lock_area(db->fd, 0L, (long)lock, I_lock_tries, I_lock_delay,
         I_lock_timeout) != I_OK)
         ERRX(I_LOCKERR)
      unlock = 1;
      }
   if (_cb_read(db->fd, 0L, sizeof(names_length), &names_length) != I_OK
       || _cb_read(db->fd, (long) sizeof(names_length),
                   sizeof(db->field_count), &db->field_count) != I_OK) {
      if (unlock) _cb_unlock_area(db->fd, 0L, (long)lock);
      SET_EC(I_IO)
      return (I_ERROR);
      }
   len = (db->field_count + 1) * sizeof(char *);
   if ((db->field_names = (char **) malloc(len)) == NULL) {
      if (unlock) _cb_unlock_area(db->fd, 0L, (long)lock);
      SET_EC(I_OOM)
      return (I_ERROR);
      }
   if ((db->field_buf = (char *) malloc(names_length)) == NULL) {
      if (unlock) _cb_unlock_area(db->fd, 0L, (long)lock);
      SET_EC(I_OOM)
      return (I_ERROR);
      }

   if ((db->field_type = (char *) malloc(db->field_count)) == NULL) {
      if (unlock) _cb_unlock_area(db->fd, 0L, (long)lock);
      SET_EC(I_OOM)
      return (I_ERROR);
      }
   if (_cb_read(db->fd, lock, names_length, db->field_buf) != I_OK) {
      if (unlock) _cb_unlock_area(db->fd, 0L, (long)lock);
      SET_EC(I_IO)
      return (I_ERROR);
      }
   if (unlock) _cb_unlock_area(db->fd, 0L, (long)lock);

   db->first_rec = sizeof(names_length)
                   + sizeof(db->field_count) + names_length;
   p = db->field_buf;

   min_rec_size = sizeof(Rec_Len);
   for (i = 0; i < db->field_count; i++) {
      if ((cptr = strchr(p, ',')) == NULL) {
         type = I_STRING;
         min_rec_size++;
         }
      else {
         cptr++;
         if (*cptr == 'S' || *cptr == 's') {
            type = I_STRING;    /* string */
            min_rec_size++;
            }
         else if (*cptr == 'C' || *cptr == 'c') {
            type = I_CSTRING;   /* case sensitive string */
            min_rec_size++;
            }
         else if (*cptr == 'I' || *cptr == 'i') {
            type = I_INTEGER;   /* integer */
            min_rec_size += sizeof(short);
            }
         else if (*cptr == 'U' || *cptr == 'u') {
            type = I_UNSIGNED;  /* unsigned */
            min_rec_size += sizeof(unsigned short);
            }
         else if (*cptr == 'L' || *cptr == 'l') {
            type = I_LONG;      /* long integer */
            min_rec_size += sizeof(long);
            }
         else if (*cptr == 'F' || *cptr == 'f') {
            type = I_FLOAT;     /* floating point */
            min_rec_size += sizeof(float);
            }
         else if (*cptr == 'D' || *cptr == 'd') {
            type = I_DOUBLE;    /* double precision */
            min_rec_size += sizeof(double);
            }
         else if (*cptr == 'B' || *cptr == 'b') {
            type = I_BINARY;    /* binary */
            cptr++;
            if (*cptr == ':') cptr++;
            length = 0;
            while (*cptr >= '0' && *cptr <= '9')
               length = 10 * length + (*cptr++ - '0');
            if (length == 0) length = 1;
            else if (length > MAX_INT) length = MAX_INT;
            min_rec_size += length;
            if (length >= 1 && db->field_len == NULL) {
               if ((db->field_len =
                  (Rec_Len *) malloc(db->field_count * sizeof(int))) == NULL)
                  ERRX(I_OOM)
               for (j = 0; j < i; j++) db->field_len [j] = 1;
               db->field_len[i] = (int) length;
               }
            else if (db->field_len != NULL) {
               db->field_len[i] = (int) length;
               }
            }
         else type = I_STRING;
         }
      db->field_type[i] = (char) type;
      db->field_names[i] = p;
      p += strlen(p) + 1;
      }
   if (min_rec_size < 0 || min_rec_size > MIN_REC_LIMIT)
      db->min_rec_size = MIN_REC_LIMIT;
   else db->min_rec_size = (int) min_rec_size;
   db->field_names[i] = NULL;
   return (I_OK);
   } /* I_rd_header */
/**/
/*
        I_ifopen - open and initialize all indexes in the index file

        Returns I_OK or I_ERROR.
*/

int I_ifopen(Db_Obj *db, char *index_file, int share, int readonly)
   /* db = OUTPUT indexes attached to this */
   /* index_file = name of index file to open */
   /* share = I_YES to open the index files shared */
   /* readonly = I_YES to open the index files for read only access */
{
   Index_Obj *index;
   int nbr;
   int desc_length;
   int status;

   if ((db->index_filename = (char *) malloc(strlen(index_file) + 1)) == NULL)
      ERRX(I_OOM)
   strcpy(db->index_filename, index_file);

   db->desc_tree = cbopen_shared(index_file, I_compare, share, readonly);
   if (db->desc_tree == NULL) ERRX(I_CORRUPTINDEX)
   db->hole_tree = cbopen_shared(index_file, I_compare, share, readonly);
   if (db->hole_tree == NULL) ERRX(I_CORRUPTINDEX)
   db->index_list = NULL;
   while ((status = I_next_index(db->desc_tree, I_db_key.key_buf,
                                 I_db_key.key_buf_len, &desc_length, &nbr))
           == I_OK) {
      index = I_iopen(db, I_db_key.key_buf, desc_length, nbr, share, readonly);
      if (index == NULL) {
         /* isam_errno already set */
         status = I_ERROR;
         break;
         }
      db->index_count++;
      add_member((void **) &db->index_list, (Any_Set *) index);
      }
   if (status == I_ERROR) {
      /* isam_errno already set */
      return (I_ERROR);
      }
   return (I_OK);
   } /* I_ifopen */
/**/
/*
        I_kludge_physical - make a kludged physical index for temp processing

        Returns I_OK or I_ERROR.
*/

static int I_kludge_physical(Db_Obj *db)
{
   Index_Obj *index;
   int i;

   if ((index = I_new_index_obj()) == NULL) return (I_ERROR);
   index->index_nbr = PHYSICAL_INDEX;
   if ((index->index_name = (char *) malloc(strlen(iphysical) + 1)) == NULL) {
      free(index);
      SET_EC(I_OOM)
      return (I_ERROR);
      }
   strcpy(index->index_name, iphysical);
   db->phys_ptr = NO_CURRENT;
   index->db = db;
   index->field_count = db->field_count;
   if ((index->field_nbrs = (int *) malloc(index->field_count * sizeof(int)))
        == NULL) {
      free(index->index_name);
      free(index);
      SET_EC(I_OOM)
      return (I_ERROR);
      }
   for (i = 0; i < index->field_count; i++) index->field_nbrs [i] = i;

   index->unique = '\0';
   index->field_segs = NULL;
   db->index_count = 1;
   add_member((void **) &db->index_list, (Any_Set *) index);
   return (I_OK);
   } /* I_kludge_physical */
/**/
/*
        I_new_index_obj - alloc and initialize new Index_Obj

        Returns pointer to initialized index object, or NULL for error.
*/

static Index_Obj *I_new_index_obj()
{
   Index_Obj *index;

   if ((index = (Index_Obj *) malloc(sizeof(Index_Obj))) == NULL) {
      SET_EC(I_OOM)
      return (NULL);
      }
   index->next = NULL;
   index->index_nbr = 0;
   index->index_name = NULL;
   index->tree = NULL;
   index->db = NULL;
   index->field_count = 0;
   index->field_nbrs = NULL;
   index->position_status = I_BOI;
   index->unique = '\0';
   index->field_segs = NULL;
   return (index);
   } /* I_new_index_obj */
/**/
/*
        I_next_index - get next index name and number from cbt file

        Returns I_OK and fills in index_buf, index_nbr, and desc_length,
        or returns I_EOI if no more indexes,
        or returns I_ERROR if an error reading the cbt file.
*/

static int I_next_index(Cbtree *tree, char *index_buf, int index_buf_len,
                        int *desc_length, int *index_nbr)
   /* tree          = tree to read from */
   /* index_buf     = OUTPUT description from tree */
   /* index_buf_len = of index_buf */
   /* desc_length   = OUTPUT length of description read */
   /* index_nbr     = OUTPUT number assigned to index */
{
   int status;
   Item item;

   if ((status = cbnext(tree, &item)) == I_OK) {
      *index_nbr = (int) item;
      if (cbkeylen(tree, desc_length) == I_ERROR
          || cbkey(tree, index_buf, index_buf_len) == I_ERROR)
         ERRX(I_CBT)
      if ((unsigned char)(*index_buf) != I_pfx_index) return (I_EOI);
      return (I_OK);
      }
   else if (status == I_EOI) return (I_EOI);
   else if (status == I_ERROR) SET_EC(I_CBT)
   return (I_ERROR);
   } /* I_next_index */
/**/
/*
        I_pull_desc - extract the index description from the key buffer

        Returns I_OK or I_ERROR, and fills in index->field_count and the
        index->field_nbrs (exactly field_count elements).
*/

static int I_pull_desc(Db_Obj *db, char *buf, int buflen, Index_Obj *index)
   /* db     = associated database */
   /* buf    = buffer to pull from */
   /* buflen = length of buf */
   /* index  = OUTPUT description items filled in */
{
   int n;

   int i, length, start;
   char *cptr, *ptr;

   buf++;                          /* skip the prepend byte */
   buflen--;
   while (buflen--, *buf++)        /* skip the index name */
      if ((*buf == 'U' || *buf == 'u') && *(buf - 1) == ',') index->unique++;
   if (index->index_nbr == PHYSICAL_INDEX) {
      index->field_count = db->field_count;
      if ((index->field_nbrs =
           (int *) malloc(index->field_count * sizeof(int))) == NULL)
         ERRX(I_OOM)
      for (n = 0; n < index->field_count; n++) index->field_nbrs[n] = n;
      return (I_OK);
      }
   index->field_count = I_count_fields(buf, buflen);
   if ((index->field_nbrs =
        (int *) malloc(index->field_count * sizeof(int))) == NULL)
      ERRX(I_OOM)
   for (n = 0; n < index->field_count; n++) {
      if ((ptr = strchr(buf, ':')) != NULL) {
         if (index->field_segs == NULL) {
            if ((index->field_segs =
                 (Seg_Info *) malloc(index->field_count * sizeof(Seg_Info)))
                 == NULL) {
               free(index->field_nbrs);
               index->field_nbrs = NULL;
               SET_EC(I_OOM)
               return (I_ERROR);
               }
            for (i = 0; i < n; i++) {
               index->field_segs[i].start = 0;
               index->field_segs[i].length = 0;
               }
            }
         start = length = 0;
         cptr = ptr++;
         while (*ptr >= '0' && *ptr <= '9')
            length = 10 * length + (*ptr++ - '0');
         if (*ptr == ',') {
            start = length;
            length = 0;
            ptr++;
            while (*ptr >= '0' && *ptr <= '9')
               length = 10 * length + (*ptr++ - '0');
            }
         if (length != 0) *cptr = '\0';
         index->field_segs [n].start = start;
         index->field_segs [n].length = length;
         }
      else if (index->field_segs != NULL) {
         index->field_segs [n].start = 0;
         index->field_segs [n].length = 0;
         }
      if ((index->field_nbrs[n] =
           I_mat_namelist(buf, db->field_names, db->field_count)) == -1)
         ERRX(I_CORRUPTINDEX)
      if (ptr != NULL) buf = ptr;
      buf += strlen(buf) + 1;
      }
   return (I_OK);
   } /* I_pull_desc */
/**/
/*
        I_count_fields - count the nbr of nul-terminated strings in buffer

        Returns number of nul-terminated strings in the buffer
*/

static int I_count_fields(char *buffer, int buflen)
   /* buffer = buffer to count */
   /* buflen = length of buffer */
{
   int nfields;

   nfields = 0;
   while (buflen-- > 0) if (*buffer++ == 0) nfields++;
   return (nfields);
   } /* I_count_fields */

/*
        I_mat_namelist - match name in buffer to a name in the list of names

        Returns the element number of the name that matches the buffer,
        or -1 if no match found.
*/

static int I_mat_namelist(char *buffer, char *names[], int nbr_of_names)
   /* buffer       = name to match */
   /* names[]      = list of names to look thru */
   /* nbr_of_names = number of names in buffer */
{
   int element_nbr;

   for (element_nbr = 0; element_nbr < nbr_of_names; element_nbr++) {
      if (stricmp(buffer, *names++) == 0) return (element_nbr);
      }
   return (-1);
   } /* I_mat_namelist */
/**/
/*
        I_compare - key compare for ISAM indexes

        Return: I_LESS    if keya <  keyb
                I_EQUAL   if keya == keyb
                I_GREATER if keya >  keyb
*/

int I_ENTRY I_compare(char *keya, int lena, char *keyb, int lenb)
{
   int status;          /* compare status */
   int field_count;     /* number of fields in key */
   int *field_nbr;      /* field number for key segment */
   char *field_type;    /* field types comprising index */
   char *keya_end;      /* end of keya */
   char *keyb_end;      /* end of keyb */
   int length;          /* key length */
   int i;               /* counter */
   static Index_Obj *index;/* pointer to index object */
   static int index_nbr;   /* index number */
   /*
           If keys have different prepend bytes, return answer.
   */

   if (*((unsigned char *) keya) < *((unsigned char *) keyb)) return (I_LESS);
   if (*((unsigned char *) keya) > *((unsigned char *) keyb)) return (I_GREATER);

   /*
            If comparing only prepend byte, return answer.
   */
   if (lena == 1) {
      if (lenb == 1) return I_EQUAL;
      else return (I_LESS);
      }
   if (lenb == 1) return I_GREATER;

   /*
           Use case insensitive compare for index description keys
           (I_pfx_index). Compare hole keys (I_pfx_offset and I_pfx_size)
           using their respective types. For all other keys, compare
           each field according to type.
   */
   i = *((unsigned char *) keya);
   keya++;
   keyb++;
   switch (i) {
      case I_PFX_INDEX:
         status = stricmp(keya, keyb);
         if (status < 0) return (I_LESS);
         if (status > 0) return (I_GREATER);
         return (I_EQUAL);
      case I_PFX_OFFSET:
         status = memcmp(keya, keyb, sizeof(Ptr));
         if (status < 0) return (I_LESS);
         if (status > 0) return (I_GREATER);
         return (I_EQUAL);
      case I_PFX_SIZE:
         status = memcmp(keya, keyb, sizeof(Rec_Len));
         if (status < 0) return (I_LESS);
         if (status > 0) return (I_GREATER);
         return (I_EQUAL);
      default:
         if (I_db_last != I_db_object || index_nbr != i) {
            I_db_last = I_db_object;
            index_nbr = i;
            index = I_db_last->index_list;
            while (index_nbr != index->index_nbr) index = index->next;
            }
         field_type = I_db_last->field_type;
         field_count = index->field_count;
         field_nbr = index->field_nbrs;
         i = 1;
         loop:
         switch (field_type [*field_nbr]) {
            case I_STRING:
               status = stricmp(keya, keyb);
               if (status < 0) return (I_LESS);
               if (status > 0) return (I_GREATER);
               if (i == field_count) return (I_EQUAL);
               length = strlen(keya) + 1;
               break;
            case I_CSTRING:
               status = strcmp(keya, keyb);
               if (status < 0) return (I_LESS);
               if (status > 0) return (I_GREATER);
               if (i == field_count) return (I_EQUAL);
               length = strlen(keya) + 1;
               break;
            case I_INTEGER:
               if (*((short *) keya) < *((short *) keyb)) return (I_LESS);
               if (*((short *) keya) > *((short *) keyb)) return (I_GREATER);
               if (i == field_count) return (I_EQUAL);
               length = sizeof(short);
               break;
            case I_UNSIGNED:
               if (*((unsigned short *) keya) < *((unsigned short *) keyb)) return (I_LESS);
               if (*((unsigned short *) keya) > *((unsigned short *) keyb)) return (I_GREATER);
               if (i == field_count) return (I_EQUAL);
               length = sizeof(unsigned short);
               break;
            case I_LONG:
               if (*((long *) keya) < *((long *) keyb)) return (I_LESS);
               if (*((long *) keya) > *((long *) keyb)) return (I_GREATER);
               if (i == field_count) return (I_EQUAL);
               length = sizeof(long);
               break;
            case I_FLOAT:
               if (*((float *) keya) < *((float *) keyb)) return (I_LESS);
               if (*((float *) keya) > *((float *) keyb)) return (I_GREATER);
               if (i == field_count) return (I_EQUAL);
               length = sizeof(float);
               break;
            case I_DOUBLE:
               if (*((double *) keya) < *((double *) keyb)) return (I_LESS);
               if (*((double *) keya) > *((double *) keyb)) return (I_GREATER);
               if (i == field_count) return (I_EQUAL);
               length = sizeof(double);
               break;
            case I_BINARY:
               if (I_db_last->field_len == NULL) {
                  if (*((unsigned char *) keya) < *((unsigned char *) keyb)) return (I_LESS);
                  if (*((unsigned char *) keya) > *((unsigned char *) keyb)) return (I_GREATER);
                  length = 1;
                  }
               else {
                  length = I_db_last->field_len [*field_nbr];
                  status = memcmp(keya, keyb, length);
                  if (status < 0) return (I_LESS);
                  if (status > 0) return (I_GREATER);
                  }
               if (i == field_count) return (I_EQUAL);
               break;
            default: length = 0;
            }
         if (i++ == 1) {
            keya_end = keya + lena - 1;
            keyb_end = keyb + lenb - 1;
            }
         keya += length;
         keyb += length;
         if (keya < keya_end) {
            if (keyb < keyb_end) {
               field_nbr++;
               goto loop;
               }
            return (I_GREATER);
            }
         if (keyb < keyb_end) return (I_LESS);
      }
   return (I_EQUAL);
   } /* I_compare */

/* end opendb.c */
