/*      Copyright (c) 1987 Shrier and Deihl. Licensed to Mix Software, Inc.
        Copyright (c) 1995 Mix Software, Inc. Multi-user version

        holes.c - ISAM INTERNAL functions to manage empty holes in the
                data file
*/
#include "isamlib.h"
#include "blockio.h"
#include <limits.h>
/*
        Defined Functions

        I_emptyrec              return address of new empty record
        I_add_hole              add unused record to hole chains
*/

/* get hole info from tree */
static int I_rd_hole_info(Cbtree *tree, Ptr *offset, Rec_Len *size);

/* merge new hole with adjacent holes */
static int I_merge_holes(Db_Obj *db, Ptr prev_ptr, Rec_Len prev_len,
                         Ptr new_ptr, Rec_Len new_len, Ptr next_ptr,
                         Rec_Len next_len);

/* I_YES or I_NO, holes are adjacent */
static int I_adjacent(Ptr left_offset, Rec_Len left_len, Ptr right_offset);

/* return size, offset of first size hole */
static int I_first_hole(Cbtree *tree, Rec_Len *hole_size, Ptr *hole_offset);

/* add hole to size chain */
static int I_add_size_hole(Cbtree *tree, Rec_Len size, Ptr offset);

/* add hole to offset chain */
static int I_add_offset_hole(Cbtree *tree, Rec_Len size, Ptr offset);

/* remove hole from size, offset chains */
static Ptr I_rm_hole(Cbtree *tree, Rec_Len hole_size, Ptr hole_offset);

/* remove hole from size chain */
static int I_rm_size_hole(Cbtree *tree, Rec_Len size, Ptr offset);

/* remove hole from offset chain */
static int I_rm_offset_hole(Cbtree *tree, Rec_Len size, Ptr offset);

/* make hole larger or smaller */
static Ptr I_mod_hole(Db_Obj *db, Rec_Len old_size, Ptr old_offset,
                      Rec_Len new_size, Ptr new_offset);

/* change size of hole */
static int I_mod_size_hole(Cbtree *tree, Rec_Len old_size, Ptr old_offset,
                           Rec_Len new_size, Ptr new_offset);

/* change offset of hole */
static int I_mod_offset_hole(Cbtree *tree, Rec_Len old_size, Ptr old_offset,
                             Rec_Len new_size, Ptr new_offset);

/* make key for size chain */
static void I_mk_size_key(Rec_Len size, char *key);

/* make key for offset chain */
static void I_mk_offset_key(Ptr offset, char *key);

/* get size from key */
static void I_get_size(char *key, Rec_Len *size);

/* get offset from key */
static void I_get_offset(char *key, Ptr *offset);

/**/
/*
        I_emptyrec - return addr of empty record of at least specified length

        If there is a hole large enough, use all or part of it;
        else add new space at end of file

        Returns address of empty record and updates length with real length of
        empty record to be used (may be 1 or 2 bytes larger than requested)
        or NULL for error
*/

Ptr I_emptyrec(Db_Obj *db, Rec_Len *length)
   /* db     = database to look at */
   /* length = OUTPUT real length of new record */
{
   Rec_Len hole_size;
   Ptr hole_offset;
   Rec_Len new_size;
   Ptr new_offset;
   int status;

   I_ASSERT((! db->share) || (db->locked & I_X_WRITELOCK) != 0);
   if (db->hole_tree != NULL) {
      status = I_first_hole(db->hole_tree, &hole_size, &hole_offset);
      if (status == I_ERROR) return (0);
      }
   else status = I_NO;
   if ((status == I_YES) && (*length <= hole_size)) {
      if (*length + db->min_rec_size > hole_size) {
         *length = hole_size;
         return (I_rm_hole(db->hole_tree, hole_size, hole_offset));
         }
      else {
         /* use part of hole; put rest of hole back in chain */
         new_size = hole_size - *length - sizeof(Rec_Len);
         new_offset = hole_offset + *length + sizeof(Rec_Len);
         if (I_mod_hole(db, hole_size, hole_offset, new_size, new_offset)
            == 0) return (0);
         return (hole_offset);
         }
      }
   /* no empty block that is large enough, find end of file */
   if (_cb_filesize(db->fd, &new_offset) != I_OK) {
      SET_EC(I_IO)
      return (0);
      }
   /* write the record length to extend the file & allow locking */
   if (_cb_write(db->fd, new_offset, sizeof(Rec_Len), &length) != I_OK) {
      SET_EC(I_IO)
      return (0);
      }
   return (new_offset);
   } /* I_emptyrec */
/**/
/*
        I_add_hole - add hole in data file to hole chains

        Update record length in data file, size chain, and offset chain.
        Try to merge new hole into adjacent existing hole first.

        Return I_OK or I_ERROR.
*/

int I_add_hole(Db_Obj *db, Ptr new_ptr, Rec_Len new_len)
   /* db      = database to look at */
   /* new_ptr = lseek pointer of discarded record */
   /* new_len = length of discarded record */
{
   char new_key[sizeof(Ptr) + sizeof(I_pfx_offset)];
   Ptr next_ptr;
   Rec_Len next_len;
   Ptr prev_ptr;
   Rec_Len prev_len;
   Item item;
   Cbtree *tree;
   int status;

   I_ASSERT((! db->share) || (db->locked & I_X_WRITELOCK) != 0);
   tree = db->hole_tree;
   I_mk_offset_key(new_ptr, new_key);
   if (cbfind(tree, new_key, sizeof(new_key), &item) == I_ERROR) ERRX(I_CBT)
   I_rd_hole_info(tree, &next_ptr, &next_len);
   if (cbprev(tree, &item) == I_ERROR) ERRX(I_CBT)
   I_rd_hole_info(tree, &prev_ptr, &prev_len);
   if ((status =
      I_merge_holes(db, prev_ptr, prev_len, new_ptr, new_len,
                    next_ptr, next_len))
      == I_ERROR) {
      return (I_ERROR);
      }
   else if (status == I_YES) ; /* new hole inserted during merge */
   else {
      if (I_putlen(db->fd, new_ptr, (Rec_Len)(-new_len)) == I_ERROR)
         return (I_ERROR);
      if (I_add_size_hole(tree, new_len, new_ptr) == I_ERROR) return (I_ERROR);
      if (I_add_offset_hole(tree, new_len, new_ptr) == I_ERROR) {
         I_rm_size_hole(tree, new_len, new_ptr);
         return (I_ERROR);
         }
      }
   return (I_OK);
   } /* I_add_hole */
/**/
/*
        I_rd_hole_info - get size, offset from current offset entry in chain

        Offset and size are set to 0 if there is no offset entry here

        Returns I_OK and fills in offset and size, or I_ERROR
*/

static int I_rd_hole_info(Cbtree *tree, Ptr *offset, Rec_Len *size)
   /* tree   = tree to read */
   /* offset = OUTPUT offset of hole */
   /* size   = OUTPUT size of hole */
{
   char key[sizeof(Ptr) + sizeof(I_pfx_offset)];
   Item item;
   int status;

   *offset = 0;
   *size = 0;
   status = cbkey(tree, key, sizeof(key));
   if (status == I_EOI || status == I_BOI) return (I_OK);
   if ((status == I_ERROR) || (cbcurr(tree, &item) == I_ERROR)) ERRX(I_CBT)
   if ((unsigned char)key[0] == I_pfx_offset) {
      I_get_offset(key, offset);
      *size = (Rec_Len) item;
      }
   return (I_OK);
   } /* I_rd_hole_info */
/**/
/*
        I_merge_holes - attempt to merge new hole with adjacent holes

        Returns I_YES and inserts new hole,
                I_NO there are no adjacent holes or merged hole will be
                too big,
                or I_ERROR for error.
*/

static int I_merge_holes(Db_Obj *db, Ptr prev_ptr, Rec_Len prev_len,
                                     Ptr new_ptr, Rec_Len new_len,
                                     Ptr next_ptr, Rec_Len next_len)
   /* db       = database to look at */
   /* prev_ptr = lseek pointer of preceding record */
   /* prev_len = length of preceding record */
   /* new_ptr  = lseek pointer of new record */
   /* new_len  = length of new record */
   /* next_ptr = lseek pointer of following record */
   /* next_len = length of following record */
{
   long newsize;
   long maxsize;

   if (sizeof(Rec_Len) == sizeof(short)) maxsize = SHRT_MAX;
   else if (sizeof(Rec_Len) == sizeof(int)) maxsize = INT_MAX;
   else maxsize = LONG_MAX;

   /* note - Rec_Len must be a signed type to allow the deleted (sign) flag */
   if (prev_len < 0 || new_len < 0 || next_len < 0) return I_NO;

   /* check for a merge of all three holes */
   if ((I_adjacent(prev_ptr, prev_len, new_ptr) == I_YES) &&
       (I_adjacent(new_ptr, new_len, next_ptr) == I_YES)) {
      newsize = (long) prev_len + (long) new_len;
      if (newsize > 0 && (newsize += (long) next_len) > 0
          && newsize <= maxsize) {
         if ((newsize += (long) (2 * sizeof(Rec_Len))) > 0
              && newsize <= maxsize) {
            if (I_mod_hole(db, prev_len, prev_ptr, (Rec_Len) newsize, prev_ptr)
                == 0)
               return I_ERROR;
            if (I_rm_hole(db->hole_tree, next_len, next_ptr) == 0)
               return (I_ERROR);
            return (I_YES);
            }  /* no size overflow */
         }
      }  /* all 3 are adjacent */

   /* try to merge with the previous hole */
   if (I_adjacent(prev_ptr, prev_len, new_ptr) == I_YES) {
      newsize = (long) prev_len + (long) new_len;
      if (newsize > 0 && (newsize += (long) sizeof(Rec_Len)) > 0
         && newsize <= maxsize) {
         if (I_mod_hole(db, prev_len, prev_ptr, (Rec_Len) newsize, prev_ptr)
             == 0)
            return (I_ERROR);
         return (I_YES);
         }
      }

   /* try to merge with the next hole */
   if (I_adjacent(new_ptr, new_len, next_ptr) == I_YES) {
      newsize = (long) new_len + (long) next_len;
      if (newsize > 0 && (newsize += (long) sizeof(Rec_Len)) > 0
         && newsize <= maxsize) {
         if (I_mod_hole(db, next_len, next_ptr, (Rec_Len) newsize, new_ptr)
             == 0)
            return (I_ERROR);
         return (I_YES);
         }
      }
   return (I_NO);
   } /* I_merge_holes */
/**/
/*
        I_adjacent - determine if left and right holes are adjacent

        Return I_YES or I_NO.
*/

static int I_adjacent(Ptr left_offset, Rec_Len left_len, Ptr right_offset)
   /* left_offset  = lseek pointer of left hole */
   /* left_len     = length of left hole */
   /* right_offset = lseek pointer of right hole */
{
   if (left_offset + left_len + (int)sizeof(Rec_Len) == right_offset)
      return (I_YES);
   else return (I_NO);
   } /* I_adjacent */
/**/
/*
        I_first_hole - return size and offset of first hole in size chain

        Returns I_YES found a hole and fills in size and offset,
                I_NO there is no hole,
                I_ERROR for error.
*/
#define BUFLEN  10      /* max (sizeof (Ptr), sizeof (Rec_Len)) + 1 */

static int I_first_hole(Cbtree *tree, Rec_Len *hole_size, Ptr *hole_offset)
   /* tree        = tree to look at */
   /* hole_size   = OUTPUT size of first hole in size chain */
   /* hole_offset = OUTPUT lseek pointer of first hole */
{
   Item item;
   int  status;
   char buf[BUFLEN];

   if ((status = cbtail(tree, &item)) == I_ERROR) ERRX(I_CBT)
   if (status == I_EOI || status == I_BOI) return (I_NO);
   cbkey(tree, buf, BUFLEN);
   if ((unsigned char)buf[0] == I_pfx_size) {
      I_get_size(buf, hole_size);
      *hole_offset = (Ptr) item;
      return (I_YES);
      }
   return (I_NO);
   } /* I_first_hole */
/**/
/*
        I_add_size_hole - add entry to hole size chain

        Returns I_OK or I_ERROR
*/

static int I_add_size_hole(Cbtree *tree, Rec_Len size, Ptr offset)
   /* tree   = tree to look at */
   /* size   = size of hole to add */
   /* offset = lseek pointer of hole to add */
{
   char key[sizeof(Rec_Len) + sizeof(I_pfx_size)];

   I_mk_size_key(size, key);
   if (cbinsert(tree, key, sizeof(key), (Item) offset, I_YES) != I_OK)
      ERRX(I_CBT)
   return (I_OK);
   } /* I_add_size_hole */
/**/
/*
        I_add_offset_hole - add entry to hole offset chain

        Returns I_OK or I_ERROR
*/

static int I_add_offset_hole(Cbtree *tree, Rec_Len size, Ptr offset)
   /* tree   = tree to look at */
   /* size   = size of hole to add */
   /* offset = lseek pointer of hole to add */
{
   char key[sizeof(Ptr) + sizeof(I_pfx_offset)];

   I_mk_offset_key(offset, key);
   if (cbinsert(tree, key, sizeof(key), (Item) size, I_NO) != I_OK)
      ERRX(I_CBT)
   return (I_OK);
   } /* I_add_offset_hole */
/**/
/*
        I_rm_hole - remove given hole from list

        Updates size chain, offset chain.
        No need to update data file since it is soon to be overwritten.

        Returns pointer to hole removed, or NULL for error.
*/

static Ptr I_rm_hole(Cbtree *tree, Rec_Len hole_size, Ptr hole_offset)
   /* tree        = tree to look at */
   /* hole_size   = size of hole to remove */
   /* hole_offset = lseek pointer of hole to remove */
{
   if ((I_rm_size_hole(tree, hole_size, hole_offset) == I_ERROR)
      || (I_rm_offset_hole(tree, hole_size, hole_offset) == I_ERROR))
      return (0);
   return (hole_offset);
   } /* I_rm_hole */
/**/
/*
        I_rm_size_hole - remove entry from hole size chain

        Returns I_OK or I_ERROR.
*/

static int I_rm_size_hole(Cbtree *tree, Rec_Len size, Ptr offset)
   /* tree   = tree to look at */
   /* size   = size of hole to remove */
   /* offset = lseek pointer of hole to remove */
{
   char key[sizeof(Rec_Len) + sizeof(I_pfx_size)];

   I_mk_size_key(size, key);
   if (cbdelete(tree, key, sizeof(key), (Item) offset) == I_ERROR)
      ERRX(I_CBT)
   return (I_OK);
   } /* I_rm_size_hole */
/**/
/*
        I_rm_offset_hole - remove entry from hole offset chain

        Returns I_OK or I_ERROR.
*/


static int I_rm_offset_hole(Cbtree *tree, Rec_Len size, Ptr offset)
   /* tree   = tree to look at */
   /* size   = size of hole to remove */
   /* offset = lseek pointer of hole to remove */
{
   char key[sizeof(Ptr) + sizeof(I_pfx_offset)];

   I_mk_offset_key(offset, key);
   if (cbdelete(tree, key, sizeof(key), (Item) size) == I_ERROR)
      ERRX(I_CBT)
   return (I_OK);
   } /* I_rm_offset_hole */
/**/
/*
        I_mod_hole - change size of given hole

        Update record length in data file, size chain, offset chain

        Returns pointer to new hole, or NULL for error
*/

static Ptr I_mod_hole(Db_Obj *db, Rec_Len old_size, Ptr old_offset,
                      Rec_Len new_size, Ptr new_offset)
   /* db         = database containing hole */
   /* old_size   = old size of hole */
   /* old_offset = old lseek pointer of hole */
   /* new_size   = new size of hole */
   /* new_offset = new lseek pointer of hole */
{
   if (I_putlen(db->fd, new_offset, (Rec_Len)(-new_size)) == I_ERROR)
      return (I_ERROR);
   if (I_mod_size_hole(db->hole_tree, old_size, old_offset,
                       new_size, new_offset) == I_ERROR) return (0);
   if (I_mod_offset_hole(db->hole_tree, old_size, old_offset,
                         new_size, new_offset) == I_ERROR) return (0);
   return (new_offset);
   } /* I_mod_hole */
/**/
/*
        I_mod_size_hole - modify entry in hole size chain

        Returns I_OK or I_ERROR
*/

static int I_mod_size_hole(Cbtree *tree, Rec_Len old_size, Ptr old_offset,
                           Rec_Len new_size, Ptr new_offset)
   /* tree       = tree to look at */
   /* old_size   = old size of hole */
   /* old_offset = old lseek pointer of hole */
   /* new_size   = new size of hole */
   /* new_offset = new lseek pointer of hole */
{
   char old_key[sizeof(Rec_Len) + sizeof(I_pfx_size)];
   char new_key[sizeof(Rec_Len) + sizeof(I_pfx_size)];

   I_mk_size_key(old_size, old_key);
   I_mk_size_key(new_size, new_key);
   if (cbmodify(tree, old_key, sizeof(old_key), old_offset,
                new_key, sizeof(new_key), new_offset, I_YES) == I_ERROR)
      ERRX(I_CBT)
   return (I_OK);
   } /* I_mod_size_hole */
/**/
/*
        I_mod_offset_hole - modify entry in hole offset chain

        Returns I_OK or I_ERROR
*/

static int I_mod_offset_hole(Cbtree *tree, Rec_Len old_size, Ptr old_offset,
                             Rec_Len new_size, Ptr new_offset)
   /* tree       = tree to look at */
   /* old_size   = old size of hole */
   /* old_offset = old lseek pointer of hole */
   /* new_size   = new size of hole */
   /* new_offset = new lseek pointer of hole */
{
   char old_key[sizeof(Ptr) + sizeof(I_pfx_offset)];
   char new_key[sizeof(Ptr) + sizeof(I_pfx_offset)];

   I_mk_offset_key(old_offset, old_key);
   I_mk_offset_key(new_offset, new_key);
   if (cbmodify(tree, old_key, sizeof(old_key), (Item) old_size,
                new_key, sizeof(new_key), (Item) new_size, I_YES) == I_ERROR)
      ERRX(I_CBT)
   return (I_OK);
   } /* I_mod_offset_hole */
/**/
/*
        I_mk_size_key - make deleted record size key

        No return value but fills in key.
*/

static void I_mk_size_key(Rec_Len size, char *key)
   /* size = size of deleted record */
   /* key  = OUTPUT buffer to hold key */
{
   Rec_Len *size_ptr;
   char *ptr;
   int i;

   *key++ = (char) I_pfx_size;
   if (I_msb_first == 0) {
      ptr = ((char *)&size) + sizeof(Rec_Len) - 1;
      for (i = 0; i < sizeof(Rec_Len); i++) *key++ = *ptr--;
      }
   else {
      size_ptr = (Rec_Len *) key;
      *size_ptr = size;
      }
   #if defined(I_NON_BYTE_INTS)
   else {
      if (sizeof(Rec_Len) == 2) {
         *key++ = (char)((size >> 8) & 0xff);
         *key   = (char)((size     ) & 0xff);
         }
      else
         *key++ = (char)((size >> 24) & 0xff);
         *key++ = (char)((size >> 16) & 0xff);
         *key++ = (char)((size >>  8) & 0xff);
         *key   = (char)((size      ) & 0xff);
         }
      }
   #endif
   return;
   } /* I_mk_size_key */
/**/
/*
        I_mk_offset_key - make deleted record offset key

        No return value but fills in key.
*/

static void I_mk_offset_key(Ptr offset, char *key)
   /* offset = lseek pointer of deleted record */
   /* key    = OUTPUT buffer to contain key */
{
   Ptr *offset_ptr;
   char *ptr;
   int i;

   *key++ = (char) I_pfx_offset;
   if (I_msb_first == 0) {
      ptr = ((char *)&offset) + sizeof(Ptr) - 1;
      for (i = 0; i < sizeof(Ptr); i++) *key++ = *ptr--;
      }
   else if (I_msb_first > 0) {
      offset_ptr = (Ptr *)key;
      *offset_ptr = offset;
      }
   #if defined(I_NON_BYTE_INTS)
   else {
      if (sizeof(Ptr) == 2) {
         *key++ = (char)((offset >> 8) & 0xff);
         *key   = (char)((offset     ) & 0xff);
         }
      else
         *key++ = (char)((offset >> 24) & 0xff);
         *key++ = (char)((offset >> 16) & 0xff);
         *key++ = (char)((offset >>  8) & 0xff);
         *key   = (char)((offset      ) & 0xff);
         }
      }
   #endif
   return;
   } /* I_mk_offset_key */
/**/
/*
        I_get_size - get deleted record size from key

        No return value, but fills in size.
*/

static void I_get_size(char *key, Rec_Len *size)
   /* key  = key to look at */
   /* size = OUTPUT size from key */
{
   Rec_Len *value_ptr;
   char *ptr;
   int i;

   if (I_msb_first == 0) {
      ptr = ((char *)size) + sizeof(Rec_Len) - 1;
      for (i = 0; i < sizeof(Rec_Len); i++) *ptr-- = *++key;
      }
   else {
      value_ptr = (Rec_Len *) (key + sizeof(char));
      *size = *value_ptr;
      }
   #if defined(I_NON_BYTE_INTS)
   else {
      if (sizeof(Rec_Len) == 2) {
         *size = ((((int) key[1]) << 8) & 0xff00) |
                  (((int) key[2])       & 0x00ff);
         }
      else
         *size = (  ((long) key[1]) << 24) |
                 ( (((long) key[2]) << 16) & 0x00ff0000L) |
                 ( (((long) key[3]) << 8 ) & 0x0000ff00L) |
                   (((long) key[4])        & 0x000000ffL);
         }
      }
   #endif
   return;
   } /* I_get_size */
/**/
/*
        I_get_offset - get deleted record offset from key

        No return value but fills in offset.
*/

static void I_get_offset(char *key, Ptr *offset)
   /* key    = key to look at */
   /* offset = OUTPUT offset from key */
{
   Ptr *value_ptr;
   char *ptr;
   int i;

   if (I_msb_first == 0) {
      ptr = ((char *)offset) + sizeof(Ptr) - 1;
      for (i = 0; i < sizeof(Ptr); i++) *ptr-- = *++key;
      }
   else {
      value_ptr = (Ptr *) (key + sizeof(char));
      *offset = *value_ptr;
      }
   #if defined(I_NON_BYTE_INTS)
   else {
      if (sizeof(Ptr) == 2) {
         *offset = ((((int) key[1]) << 8) & 0xff00) |
                    (((int) key[2])       & 0x00ff);
         }
      else
         *offset = (  ((long) key[1]) << 24) |
                   ( (((long) key[2]) << 16) & 0x00ff0000L) |
                   ( (((long) key[3]) << 8 ) & 0x0000ff00L) |
                     (((long) key[4])        & 0x000000ffL);
         }
      }
   #endif
   return;
   } /* I_get_offset */

/* end holes.c */
