/*      Copyright (c) 1987 Shrier and Deihl. Licensed to Mix Software, Inc.
        Copyright (c) 1995 Mix Software, Inc. Multi-user version

        addrec.c - ISAM iaddrec and supporting functions
*/
#include <stdlib.h>
#include <string.h>
#include "isamlib.h"
#include "blockio.h"

/*      Defined Functions (INTERNAL functions for ISAM internal use ONLY

        iaddrec                 add record to database and update all indexes
        I_adddata               INTERNAL add record to data file
        I_putrec                INTERNAL write record in data file
        I_putlen                INTERNAL write record length in data file
        I_calc_fieldlens        INTERNAL calculate lengths of fields
        I_calc_reclen           INTERNAL calculate length of entire record
*/
/**/
/*
        iaddrec - add new record to database and update all indexes

        If index is NULL, the current record is not changed.  If an index
        handle is specified, the current record of this index ONLY is set to
        the record just added.  The currency of other indexes is not changed.

        The list of field pointers need not be terminated by a NULL pointer,
        but it won't hurt.

        Duplicate keys are allowed, but duplicate records not detected.

        Returns I_OK or I_ERROR.
*/

I_EXPORT int I_ENTRY iaddrec(Db_Obj *db, Index_Obj *index, char *fields[])
  /* db     = database to add record to */
  /* index  = if not NULL, make new rec current in this index */
  /* fields = record to add */
{
   int status;
   Ptr cur_ptr;
   int unlock = 0;

   I_db_object = db;
   SET_EC(I_NOERR)
   if (!is_member((void **) &I_db_objs, (Any_Set *) db)) {
      SET_EC(I_NOTDB)
      status = I_ERROR;
      }
   if (db->readonly) {
      SET_EC(I_NOMODIFY)
      return (I_ERROR);
      }
   if (db->share && (db->locked & I_X_WRITELOCK) == 0) {  /* write lock idx */
      if (I_lockfile(db, I_A_WRITELOCK) != I_OK) {
         SET_EC(I_LOCKERR)
         return (I_ERROR);
         }
      unlock = 1;
      }
   status = I_adddata(db, fields, &cur_ptr);
   if (status == I_OK) {
      status = I_upindexes(db, fields, cur_ptr);
      }
   if (index && (status == I_OK)) {
      index->position_status = I_OK;
      if (index->index_nbr == PHYSICAL_INDEX) {
         db->phys_ptr = cur_ptr;
         }
      else {
         I_db_key.key_len =
            I_mkkey(index, fields, I_db_key.key_buf, I_db_key.key_buf_len);
         if (I_db_key.key_len == -1) {
            /* isam_errno already set */
            status = I_ERROR;
            }
         else {
            if (cbfinditem(index->tree, I_db_key.key_buf, I_db_key.key_len,
                           (Item *) &cur_ptr)
                != I_FOUND) status = I_ERROR;
            }
         }
      }
   if (unlock) {
      if (_cb_commitblk(db->fd) != I_OK) {
         if (status != I_ERROR) {
            status = I_ERROR;
            SET_EC(I_IO)
            }
         }
      I_unlockfile(db, I_A_WRITELOCK);
      }
   return (status);
   } /* iaddrec */
/**/
/*
        I_adddata - add a record to the datafile

        Returns I_OK and fills in data_ptr with where the record was added,
        or returns I_ERROR.
*/

int I_adddata(Db_Obj *db, char *fields[], Ptr *data_ptr)
   /* db       = database to add to */
   /* fields   = record to add */
   /* data_ptr = OUTPUT addr where record added */
{
   Rec_Len rec_length;
   Ptr cur_ptr;
   int status;
   int *field_lengths;
   int unlock = 0;

   field_lengths = (int *) malloc(db->field_count * sizeof(int *));
   if (field_lengths == NULL) ERRX(I_OOM)

   I_calc_fieldlens(fields, db, field_lengths);
   rec_length = I_calc_reclen(field_lengths, db->field_count);
   status = I_OK;

   /* find an empty record */
   if ((cur_ptr = I_emptyrec(db, &rec_length)) == 0)
      /* isam_errno already set */
      status = I_ERROR;
   if (status == I_OK) status = I_liblock(db, cur_ptr, &unlock);
   if (status == I_OK) {
      if ((I_putrec(db, cur_ptr, rec_length, db->field_count,
                      fields, field_lengths) != I_OK) ) {
         /* isam_errno already set */
         status = I_ERROR;
         }
      }
   if (unlock) I_libunlock(db);
   if (status == I_OK) *data_ptr = cur_ptr;
   free(field_lengths);
   return (status);
   } /* I_adddata */
/**/
/*
        I_putrec - write data record to file
                   If the file is shared, the record should already
                   be locked.
        Returns I_OK or I_ERROR.
*/

int I_putrec(Db_Obj *db, Ptr cur_ptr, Rec_Len rec_length, int field_count,
             char *fields[], int field_lengths[])
   /* db            = database to write to */
   /* cur_ptr       = lseek pointer, where to write the data */
   /* rec_length    = number of bytes to write */
   /* field_count   = number of fields in record */
   /* fields        = the record to add */
   /* field_lengths = length of each field in fields array */
{
   int i;
#if BUFFERED_WRITE
   static unsigned char bufr[512];
   int used;
#endif

   I_ASSERT(! db->share || I_locktest(db, cur_ptr) == I_YES);

/* JTS 05/13/1997 - buffered write to improve speed */
#if BUFFERED_WRITE
   memcpy(bufr, (unsigned char *)&rec_length, sizeof(Rec_Len));
   used = sizeof(Rec_Len);
   for (i = 0; i < field_count; i++) {
      if ((int)sizeof(bufr) - used >= field_lengths[i]) {
         memcpy(&bufr[used], fields[i], field_lengths[i]);
         used += field_lengths[i];
         }
      else {
         if (used > 0) {
            if (_cb_write(db->fd, cur_ptr, used, bufr) != I_OK) ERRX(I_IO)
            cur_ptr += used;
            used = 0;
            }
         if (field_lengths[i] >= sizeof(bufr)) {
            if (_cb_write(db->fd, cur_ptr, field_lengths[i], fields[i]) != I_OK)
               ERRX(I_IO)
            cur_ptr += field_lengths[i];
            }
         else {
            memcpy(bufr, fields[i], field_lengths[i]);
            used = field_lengths[i];
            }
         }
      }
   if (used > 0) {
      if (_cb_write(db->fd, cur_ptr, used, bufr) != I_OK) ERRX(I_IO)
      }
#else
   if (I_putlen(db->fd, cur_ptr, rec_length) == I_ERROR) return (I_ERROR);
   cur_ptr += sizeof(Rec_Len);
   for (i = 0; i < field_count; i++) {
      if (_cb_write(db->fd, cur_ptr, field_lengths[i], fields[i]) != I_OK)
         ERRX(I_IO)
      cur_ptr += field_lengths[i];
      }
#endif
   return (I_OK);
   } /* I_putrec */
/**/
/*
        I_putlen - write length of data record to file

        Returns I_OK or I_ERROR.
*/

int I_putlen(int fd, Ptr cur_ptr, Rec_Len length)
   /* fd      = file to write to */
   /* cur_ptr = lseek pointer, where to write the length */
   /* length  = value to write */
{
   if (_cb_write(fd, cur_ptr, sizeof(Rec_Len), &length) != I_OK) ERRX(I_IO)
   return (I_OK);
   } /* I_putlen */
/**/
/*
        I_calc_fieldlens - calculate lengths of each field in fields and
                fill in values in lengths (includes length of terminating
                nul of each field)

        Returns nothing, but updates lengths array
*/

void I_calc_fieldlens(char *fields[], Db_Obj *db, int lengths[])
   /* fields  = the fields to calculate */
   /* db      = pointer to database object */
   /* lengths = OUTPUT length of each field */
{
   int i, field_count;
   char *type;

   field_count = db->field_count;
   type = db->field_type;
   for (i = 0; i < field_count; i++) {
      switch (type[i]) {
         case I_STRING:
         case I_CSTRING:
            lengths[i] = strlen(fields[i]) + 1;
            continue;
         case I_INTEGER:
            lengths[i] = sizeof(short);
            continue;
         case I_UNSIGNED:
            lengths[i] = sizeof(unsigned short);
            continue;
         case I_LONG:
            lengths[i] = sizeof(long);
            continue;
         case I_FLOAT:
            lengths[i] = sizeof(float);
            continue;
         case I_DOUBLE:
            lengths[i] = sizeof(double);
            continue;
         case I_BINARY:
            if (db->field_len == NULL) lengths[i] = 1;
            else lengths[i] = db->field_len[i];
            continue;
         }
      }
   return;
   } /* I_calc_fieldlens */

/*
        I_calc_reclen - calculate length of entire record, given lengths
                of each field

        Returns length of entire record
                (includes terminating nuls of each field)
*/


int I_calc_reclen(int field_lengths[], int nbr_fields)
   /* field_lengths = previously calculated lengths */
   /* nbr_fields    = number of values in field_lengths */
{
   int i;
   int length;

   for (length = 0, i = 0; i < nbr_fields; i++) length += field_lengths[i];
   return (length);
   } /* I_calc_reclen */

/* end addrec.c */
