/*      Copyright (c) 1995 Mix Software, Inc.

        MSWINDOWS version
        blockio.c - module to read and write blocks from an open file
                (internal use only)

        The file is assumed to be a collection of blksize-sized blocks.
        The buffer must be large enough to hold blksize bytes.
*/

#include <windows.h>
#ifndef NO_WINDOWS_H_FILE

#include <dos.h>
#include <stdlib.h>
#include "cdtdef.h"
#include "blockio.h"

/* define 32 bit registers for int86 */
#if defined(M_I386) || defined(__386__)
   #define AX x.eax
   #define BX x.ebx
   #define CX x.ecx
   #define DX x.edx
   #define SI x.esi
   #define DI x.edi
   #define int86 int386
   #define SET_FS(s) s.fs = s.ds
   #define SET_GS(s) s.gs = s.ds
#else
   #define AX x.ax
   #define BX x.bx
   #define CX x.cx
   #define DX x.dx
   #define SI x.si
   #define DI x.di
   #define SET_FS(s)
   #define SET_GS(s)
#endif

#define UPPER_WORD(value) ((unsigned)((value) >> 16));
#define LOWER_WORD(value) ((unsigned)((value) & 0xffff));

/*
        Defined functions (CBT internal use only)

        cbw_openfile               open a random file
        cbw_close                  close a file
        cbw_read                   read a random area from a file
        cbw_readblk                read block from file
        cbw_readseq                read sequentially
        cbw_write                  write to a file
        cbw_writeblk               write block to file
        cbw_writeseq               write sequentially
        cbw_commitblk              commit updates to disk
        cbw_lock_area              lock a portion of a file
        cbw_unlock_area            unlock a portion of a file
        cbw_delay                  delay (before lock retries)
        cbw_prints                 print a string to stdout
        cbw_can_share              can files be shared?
*/
int cbw_read(int fd, unsigned long location, int size, void *buffer);
int cbw_write(int fd, unsigned long location, int size, void *buffer);

/**/
/* ------------------------------------------------------------------------
        cbw_openfile - open a block oriented file

        Filename is the name of the file to be opened.  Share is either I_YES
        or I_NO, (1 or 0).  If share is I_YES, the file is opened for shared
        access so that multiple programs can access the file at the same time.

        Return a non-negative integer on success or -1 on error.  In the
        event of an error, errno is also set.
*/

int cbw_openfile(char *filename, int mode, int share)
{
   union REGS r;
   struct SREGS sr;
   int modeflg;

   r.h.ah = 0x3d;
   if (mode == CBT_OPEN_RDONLY) modeflg = 0x80; else modeflg = 0x82;
   if (share) modeflg |= 0x40;  /* deny none */
   else modeflg |= 0x10;        /* deny all */
   r.h.al = (unsigned char) modeflg;
   segread(&sr);
   if (sizeof(char *) == sizeof(unsigned)) r.DX = (unsigned) filename;
   else {
      r.DX = FP_OFF(filename);
      sr.ds = FP_SEG(filename);
      }
   intdosx(&r, &r, &sr);
   if (r.x.cflag != 0) {
      _cbt_doserrno = r.h.al;
      return -1;
      }
   return r.AX;
} /* cbw_openfile */
/**/
/* ------------------------------------------------------------------------
        cbw_create - create a file and open it for exclusive write

        Filename is the name of the file to be opened.
*/

int cbw_create(char *filename)
{
   union REGS r;
   struct SREGS sr;

   r.h.ah = 0x3c;
   r.CX = 0;
   segread(&sr);
   if (sizeof(char *) == sizeof(unsigned)) r.DX = (unsigned) filename;
   else {
      r.DX = FP_OFF(filename);
      sr.ds = FP_SEG(filename);
      }
   intdosx(&r, &r, &sr);
   if (r.x.cflag != 0) {
      _cbt_doserrno = r.h.al;
      return -1;
      }
   return r.AX;
} /* cbw_create */
/**/
/* ------------------------------------------------------------------------

        cbcloseblk - close a block oriented file

        Return 0 on success or -1 on error.
*/

int cbw_closeblk(int fd)
{
   union REGS r;
   r.h.ah = 0x3e;
   r.BX = fd;
   intdos(&r, &r);
   if (r.x.cflag != 0) {
      _cbt_doserrno = r.h.al;
      return -1;
      }
   return 0;
} /* cbw_closeblk */
/**/
/* ------------------------------------------------------------------------

        cbreadblk - read one block of blksize from file into buffer

        Return OK or ERROR.
*/
int cbw_readblk(int fd, Blk_Nbr blocknbr, int blksize, void *buffer)
{
   return cbw_read(fd, blocknbr * blksize, blksize, buffer);
   }

/* ------------------------------------------------------------------------

        cbw_read  - random read

        Return OK or ERROR.
*/

int cbw_read(int fd, unsigned long location, int size, void *buffer)
{
   union REGS r;
   struct SREGS sr;

   r.h.ah = 0x42;                /* seek to position */
   r.h.al = 0;
   r.BX = fd;
   r.CX = UPPER_WORD(location);
   r.DX = LOWER_WORD(location);
   intdos(&r, &r);
   if (r.x.cflag != 0) {
      _cbt_doserrno = r.h.al;
      return I_ERROR;
      }

   r.h.ah = 0x3f;                /* read file */
   r.BX = fd;
   r.CX = size;
   segread(&sr);
   if (sizeof(char *) == sizeof(unsigned)) r.DX = (unsigned) buffer;
   else {
      r.DX = FP_OFF(buffer);
      sr.ds = FP_SEG(buffer);
      }
   intdosx(&r, &r, &sr);
   if (r.x.cflag != 0) {
      _cbt_doserrno = r.h.al;
      return I_ERROR;
      }
   if (r.AX < (unsigned) size) {
      _cbt_doserrno = EC_DOS_READERR;
      return I_ERROR;
      }
   return I_OK;
   } /* cbreadblk */
/* ------------------------------------------------------------------------

        cbw_readseq  - sequential read

        Return OK or ERROR.
*/

int cbw_readseq(int fd, int size, void *buffer)
{
   union REGS r;
   struct SREGS sr;

   r.h.ah = 0x3f;                /* read file */
   r.BX = fd;
   r.CX = size;
   segread(&sr);
   if (sizeof(char *) == sizeof(unsigned)) r.DX = (unsigned) buffer;
   else {
      r.DX = FP_OFF(buffer);
      sr.ds = FP_SEG(buffer);
      }
   intdosx(&r, &r, &sr);
   if (r.x.cflag != 0) {
      _cbt_doserrno = r.h.al;
      return I_ERROR;
      }
   if (r.AX < (unsigned) size) {
      _cbt_doserrno = EC_DOS_READERR;
      return I_ERROR;
      }
   return I_OK;
   } /* cbreadblk */
/**/
/* ------------------------------------------------------------------------

        cbwriteblk - write one block of blksize to file from buffer

        Return OK or ERROR.
*/

int cbw_writeblk(int fd, Blk_Nbr blocknbr, int blksize, void *buffer)
{
   return cbw_write(fd, blocknbr * blksize, blksize, buffer);
   }

/* ------------------------------------------------------------------------

        cbw_write - random write

        Return OK or ERROR.
*/

int cbw_write(int fd, unsigned long location, int size, void *buffer)
{
   union REGS r;
   struct SREGS sr;

   r.h.ah = 0x42;                /* seek to position */
   r.h.al = 0;
   r.BX = fd;
   r.CX = UPPER_WORD(location);
   r.DX = LOWER_WORD(location);
   intdos(&r, &r);
   if (r.x.cflag != 0) {
      _cbt_doserrno = r.h.al;
      return I_ERROR;
      }

   r.h.ah = 0x40;                /* write to file */
   r.BX = fd;
   r.CX = size;
   segread(&sr);
   if (sizeof(char *) == sizeof(unsigned)) r.DX = (unsigned) buffer;
   else {
      r.DX = FP_OFF(buffer);
      sr.ds = FP_SEG(buffer);
      }
   intdosx(&r, &r, &sr);
   if (r.x.cflag != 0) {
      _cbt_doserrno = r.h.al;
      return I_ERROR;
      }
   if (r.AX < (unsigned) size) {
      _cbt_doserrno = EC_DOS_WRITERR;
      return I_ERROR;
      }
   return I_OK;
   } /* cbw_write */

/* ------------------------------------------------------------------------

        cbw_writeseq - sequential write

        Return OK or ERROR.
*/

int cbw_writeseq(int fd, int size, void *buffer)
{
   union REGS r;
   struct SREGS sr;

   r.h.ah = 0x40;
   r.BX = fd;
   r.CX = size;
   segread(&sr);
   if (sizeof(char *) == sizeof(unsigned)) r.DX = (unsigned) buffer;
   else {
      r.DX = FP_OFF(buffer);
      sr.ds = FP_SEG(buffer);
      }
   intdosx(&r, &r, &sr);
   if (r.x.cflag != 0) {
      _cbt_doserrno = r.h.al;
      return I_ERROR;
      }
   if (r.AX < (unsigned) size) {
      _cbt_doserrno = EC_DOS_WRITERR;
      return I_ERROR;
      }
   return I_OK;
   } /* cbw_write */

/**/
/* ------------------------------------------------------------------------

        cbw_remove - delete a file

        Return OK or ERROR.
*/

int cbw_remove(char *filename)
{
   union REGS r;
   struct SREGS sr;

   r.h.ah = 0x41;
   segread(&sr);
   if (sizeof(char *) == sizeof(unsigned)) r.DX = (unsigned) filename;
   else {
      r.DX = FP_OFF(filename);
      sr.ds = FP_SEG(filename);
      }
   intdosx(&r, &r, &sr);
   if (r.x.cflag != 0) {
      _cbt_doserrno = r.h.al;
      return I_ERROR;
      }
   return I_OK;
   } /* cbw_remove */

/**/
/* ------------------------------------------------------------------------

        cbw_lock_area - put a record lock on a file region

        Return OK or ERROR.
*/

int cbw_lock_area(int fd, unsigned long location, long size, int tries,
   long delay, long timeout)
{
   union REGS regs, r2;
   DWORD  stoptime;
   DWORD  time, lasttime;
   MSG msg;

   /* set registers for lock system call */
   regs.h.ah = 0x5c;
   regs.h.al = 0;
   regs.BX = fd;
   regs.CX = UPPER_WORD(location);
   regs.DX = LOWER_WORD(location);
   regs.SI = UPPER_WORD(size);
   regs.DI = LOWER_WORD(size);
   /* try to lock */
   intdos(&regs, &r2);
   if (r2.x.cflag == 0) return (I_OK);

   /* lock failed - retry based on arguments */
   if (tries < 2) return I_ERROR;
   time = GetTickCount();
   stoptime = time + timeout;

   while (tries-- > 0) {
      lasttime = time;
      while (time < stoptime && (time-lasttime) < (DWORD) delay) {
         PeekMessage(&msg, NULL, 0, 0, PM_NOREMOVE);
         time = GetTickCount();
         }
      intdos(&regs, &r2);  /* retry the lock */
      if (r2.x.cflag == 0) return (I_OK);
      if (time > stoptime) break;    /* reached timeout */
      }  /* tries */

   return (I_ERROR);
} /* cbw_lock_area */
/**/
/* ------------------------------------------------------------------------

        cbw_unlock_area - remove a record lock from a file region

        Return OK or ERROR.
*/

int cbw_unlock_area(int fd, unsigned long location, long size)
{
   union REGS regs;

   regs.h.ah = 0x5c;
   regs.h.al = 1;
   regs.BX = fd;
   regs.CX = UPPER_WORD(location);
   regs.DX = LOWER_WORD(location);
   regs.SI = UPPER_WORD(size);
   regs.DI = LOWER_WORD(size);
   intdos(&regs, &regs);
   if (regs.x.cflag != 0) return I_ERROR;
   return (I_OK);
} /* cbw_unlock_area */
/**/
/* ------------------------------------------------------------------------

        cbw_commitblk -   flush any system buffer associated with specified
                         file descriptor

        This subroutine exists to get around a serious DOS design flaw.  When
        a disk file is flushed, the data is written to the disk but neither
        the file control block (FCB) nor the file allocation table (FAT) are
        updated.  The data is on the disk but the file system doesn't know
        it is there.  In DOS 3, they added a "commit file" DOS function to
        get around this problem.  Unfortunately, not everyone is running DOS
        3 or later.  If a version less than 3.3 is found, this routine
        duplicates the file handle and close the duplicate.  This is somewhat
        slower than using the commit file function but has the benefit of
        working on all versions of DOS.

        Return OK or ERROR.
*/

int cbw_commitblk(int fd)
{
   union REGS regs;
   if (_osmajor >= 3) {
      regs.h.ah = 0x68;
      regs.BX = fd;
      intdos(&regs, &regs);  /* flush system buffer */
      if (regs.x.cflag != 0 && regs.h.ah != 0x68) return I_ERROR;
      }
   else {
      regs.h.ah = 0x45;
      regs.BX = fd;
      intdos(&regs, &regs);
      if (regs.x.cflag != 0) {
         _cbt_doserrno = regs.AX;
         return (I_ERROR);
         }
      regs.BX = regs.AX;      /* new file handle */
      regs.h.ah = 0x3e;
      intdos(&regs, &regs);
      if (regs.x.cflag != 0) {
         _cbt_doserrno = regs.AX;
         return (I_ERROR);
         }
      }
   return I_OK;
   }
/**/
/* ------------------------------------------------------------------------

        cbw_delay - time delay before lock retry

        always returns OK
*/

int cbw_delay(long time)
{
   DWORD  stoptime;
   DWORD  curtime;
   MSG msg;

   stoptime = GetTickCount() + time;
   do {
      PeekMessage(&msg, NULL, 0, 0, PM_NOREMOVE);
      curtime = GetTickCount();
      } while (curtime < stoptime);
   return (I_OK);
} /* cbw_delay */

/**/
/* ------------------------------------------------------------------------

        cbw_prints - print a string to stdout

*/

int I_ENTRY cbw_prints(const char *string)
{
   int _cbw_print(const char *string);
   return _cbw_print(string);
} /* cbw_prints */
/**/

/* ------------------------------------------------------------------------

        cbw_can_share - can files be shared?

        Return I_YES or I_NO.
*/

int cbw_can_share(void)
{
   union REGS r;
   static int share = -1;
   if (share == -1) {
      r.AX = 0x1000;
      int86(0x2f, &r, &r);
      if (r.h.al != 0xff) return 0;
      /* when running under MS Windows with share not loaded, windows  */
      /* reports that share is loaded, but file sharing does not work. */
      /* Test by attempting to unlock a block in stdin.  If share is   */
      /* not functioning, an error code of 1 (invalid function number) */
      /* is returned.  If share is present, it returns error code 0x21 */
      /* (no such lock) or 0x06 (invalid handle). */
      r.AX = 0x5c01;
      r.BX = 1;
      r.CX = r.DX = r.SI = r.DI = 0;
      int86(0x21, &r, &r);
      share = (r.h.al == 1) ? I_NO : I_YES;
      }
   return share;
} /* cbw_can_share */
/**/

/* ------------------------------------------------------------------------

        cbw_filesize - determine the size of an open file

        Return OK or ERROR.
*/

int cbw_filesize(int fd, long *size)
{
   long pos;                    /* position of block */
   union REGS r;

   r.h.ah = 0x42;                /* seek to position */
   r.h.al = 0x02;
   r.BX = fd;
   r.CX = 0;
   r.DX = 0;
   intdos(&r, &r);
   if (r.x.cflag != 0) {
      _cbt_doserrno = r.h.al;
      return I_ERROR;
      }
   if (sizeof(unsigned) > 2) pos = ((r.DX & 0xffff) << 16) | (r.AX & 0xffff);
   else pos = ((long)r.DX << 16) | (long)r.AX;
   *size = pos;
   return (I_OK);
   } /* cbw_filesize */

/* ------------------------------------------------------------------------

        _cbw_print - default print function
         used if the program has not set the print pointer.  Displays
         the text as a message box.

        Return OK or ERROR.
*/

int _cbw_print(const char *string)
{
   MessageBox(NULL, string, "C/Database Toolchest",
              MB_ICONINFORMATION | MB_OK | MB_TASKMODAL);
   return I_OK;
   } /* _cbw_print */

#endif
/* end winio.c */
