/*      Copyright (c) 1995 Mix Software, Inc.

        ctlrec.c - CBT control record module (INTERNAL)
*/
#include <stdlib.h>
#include <string.h>
#include "btree.h"
#include "blockio.h"

/*
        Defined Functions (These are all INTERNAL to CBT)

        cb_set_flush            set file buffer flush flags
        B_ctlalloc              alloc and init Ctlrec
        B_ctlfind               find a Ctlrec by filename
        B_ctlfree               free a Ctlrec
        B_ctlload               read Ctlrec from index file
        B_ctlsave               save Ctlrec back to disk
        B_ctlmake               make new Ctlrec in new index file
        B_is_ctl_valid          is this a valid control record ?
        B_calc_parity           calculate parity on buffer
        B_read_lock             lock the btree for reading
        B_write_lock            lock the btree for writing
        B_ctl_sync              synchronize memory with disk file
        B_ctl_resync            synchronize disk file with memory
        B_ctl_unlock            unlock the control record if auto locked
*/

#define B_is_read_lock(ctl) ((ctl->lock_state & CTL_READ_LOCK) != 0)
#define B_is_write_lock(ctl) ((ctl->lock_state & CTL_WRITE_LOCK) != 0)
#define B_is_locked(ctl) (ctl->lock_state != 0)
#define MANUAL_WRITE_LOCK(ctl) ((ctl->lock_state & CTL_WLOCK) == CTL_MANUAL_WRITE_LOCK)
#define MULTI_LOCK_TRIES  5     /* retries when locking two areas at once */

/**/

extern int _b_no_flush;
static Ctlrec *valid_ctl = NULL;      /* list of valid control records */
static int parity_words = 0;
static B_delay_sync = 1;
/**/
/*
        B_ctlalloc - Allocate and initialize a new Ctlrec

        Inits only parts not read from disk; other parts init'ed by bctlload.
        Returns pointer to control record or NULL if error.
*/


Ctlrec *B_ctlalloc(char *filename, int share)
{
   Ctlrec *ctl;                   /* new control record */

   if ((ctl = (Ctlrec *) malloc(sizeof(Ctlrec) + strlen(filename) + 1))
      == (Ctlrec *) NULL) {
      SET_EC(EC_NOMEM)
      SET_CC(CC_NEWCTL)
      return (NULL);
      }
   B_ctlmake(ctl, 0);
   ctl->next = valid_ctl;
   valid_ctl = ctl;
   ctl->nopen = 0;
   ctl->share = share;
   ctl->lock_state = 0;
   ctl->read_lock = sizeof(Ctldisk);
   ctl->bfile = NULL;
   ctl->update = 0L;
   ctl->current = 0;
   ctl->modified = 0;
   ctl->fname = (char *) &ctl [1];
   strcpy(ctl->fname, filename);
   if (parity_words == 0) {
      parity_words = (int)((char *)(&ctl->disk.vparity) - (char *)(&ctl->disk))
         / sizeof(unsigned short);

      }
   return (ctl);
   } /* B_ctlalloc */
/**/
/*
        B_ctlfind - find a control record by filename

        Returns pointer to control record if found, NULL if not found.
*/


Ctlrec *B_ctlfind(char *filename)
{
   Ctlrec *ctl;                   /* pointer to control record */

   for (ctl = valid_ctl; ctl; ctl = ctl->next) {
      if (strcmp(filename, ctl->fname) == 0) return (ctl);
      }
   return (NULL);
   } /* B_ctlfind */
/**/
/*
        B_ctlfree - free a control record

        Returns OK or ERROR.
*/


int B_ctlfree(Ctlrec *ctl)
{
   Ctlrec *c;                     /* for loop control rec pointer */

   if (B_is_ctl_valid(ctl) == I_NO) return (I_ERROR);
   if (valid_ctl == ctl) valid_ctl = ctl->next;
   else {
      for (c = valid_ctl; c; c = c->next) {
         if (c->next && c->next == ctl) {
            c->next = ctl->next;
            break;
            }
         }
      }
   free(ctl);
   return (I_OK);
   } /* B_ctlfree */
/**/
/*
        B_ctlload - read a control record from disk

        Inits only parts read from disk. Other parts must already by init'd
        by B_ctlalloc.  Reads only the part of the record that contains valid
        data, not the whole block.  This avoids accessing the read locks.

        Returns OK or ERROR.
*/


int B_ctlload(Ctlrec *ctl)
{
   Ctldisk buf;                 /* block 0 buffer */
   int fd;                      /* file descriptor */
   int locked = 0;              /* lock done in this function flag */

   #if EXTRA_CHECKING
   if (B_is_ctl_valid(ctl) == I_NO) {
      SET_CC(CC_READCTL)
      return (I_ERROR);
      }
   #endif
   if (ctl->share && (! B_is_locked(ctl))) {
      if (B_read_lock(ctl) != I_OK) {
         SET_CC(CC_READCTL)
         return I_ERROR;
         }
      ctl->lock_state |= CTL_AUTO_READ_LOCK;
      locked = 1;
      }
   fd = ctl->bfile->fd;
   /* The control record is always in block 0 */
   if (_cb_read(fd, 0L, sizeof(Ctldisk), &buf) != I_OK)
      ERR2X(EC_GETBLK, CC_READCTL)
   if (B_calc_parity((unsigned short *) &buf, parity_words+1) != 0)
      ERR2X(EC_PARCHECK, CC_READCTL)
   #if STRUCT_ASSIGN
   ctl->disk = buf;
   #else
   memcpy(ctl->disk,buf,sizeof(Ctldisk));
   #endif
   ctl->update = ctl->disk.update;
   ctl->current = 1;
   if (locked) {
      if (B_ctl_unlock(ctl) == I_ERROR) ERR2X(CC_READCTL, EC_UNLOCK)
      }
   return (I_OK);
   } /* B_ctlload */
/**/
/*
        B_ctlsave - save a control record to disk

        Returns OK or ERROR.
*/

int B_ctlsave(Ctlrec *ctl)
{

   CB_ASSERT(B_is_write_locked(ctl));
   if (B_is_ctl_valid(ctl) == I_NO) {
      SET_CC(CC_WRITECTL)
      return (I_ERROR);
      }
   ctl->disk.vparity = B_calc_parity((unsigned short *) ctl, parity_words);
   if (_cb_writeblk(ctl->bfile->fd, (Blk_Nbr) 0, sizeof(Ctldisk), &ctl->disk)
      != I_OK) ERR2X(EC_PUTBLK, CC_WRITECTL)
   return (I_OK);
   } /* B_ctlsave */
/**/
/*
        B_ctlmake - make a new control record
        Returns OK or ERROR.
*/

int B_ctlmake(Ctlrec *ctl, int blksize)
{
   ctl->disk.blksize = (short) blksize;
   ctl->disk.root = 0;
   ctl->disk.highblock = 0;
   ctl->disk.lowleaf = 0;
   ctl->disk.highleaf = 0;
   ctl->disk.freechain = 0;
   ctl->disk.depth = 0;
   ctl->disk.vparity = 0;
   ctl->disk.ver = CBT_FILE_VERSION;
   ctl->disk.update = 0;
   ctl->disk.keycount = 0;
   ctl->disk.vparity = B_calc_parity((unsigned short *) ctl, parity_words);
   return (I_OK);
   } /* B_ctlmake */
/**/
/*
        B_is_ctl_valid - is this a valid control record? I_YES or I_NO

        Returns I_YES or I_NO.
*/


int B_is_ctl_valid(Ctlrec *ctl)
{
   Ctlrec *p;                     /* current control record */

   for (p = valid_ctl; p; p = p->next) if (p == ctl) return (I_YES);
   SET_EC(EC_BADCTL)
   SET_CC(CC_ARGCHECK)
   return (I_NO);
   } /* B_is_ctl_valid */
/**/
/*
        B_calc_parity - calculate vertical parity over an array of words

        Returns 16 bit vertical parity
*/


unsigned short B_calc_parity(unsigned short *buf, int word_count)
{
   register int i;      /* for loop index */
   register unsigned short parity; /* calculated vertical parity */

   parity = 0;
   for (i = 0; i < word_count; i++) parity ^= *buf++;
   return (parity);
   } /* B_calc_parity */

/**/
/*
        B_ctl_sync - synchronize memory with disk file

        This routine checks the disk version of the control record against
        the memory version.  If they are the same, then the buffer pool
        does not need to be reloaded before performing the CBT operation.
        If they are not the same, the memory version is updated and the
        buffer pool is purged of all buffers containing nodes from this
        file.  This forces all accesses to get fresh copies of the data
        from the disk.

        Returns I_OK, CBT_SYNC, or I_ERROR.  I_OK means that the
        buffer pool and the file are still in sync.  CBT_SYNC means that
        the buffer pool had to be purged so that it can re-sync with the
        disk file.  Also the current record and mark should be checked for
        validity.  I_ERROR indicates an error condition.

        On exit OK, the control record is read locked.  B_ctl_unlock() must
        be called when the operation is finished.

*/

int B_ctl_sync(Ctlrec *ctl)
{
   Ctldisk rec;

   #if EXTRA_CHECKING
   if (B_is_ctl_valid(ctl) == I_NO) {
      SET_CC(CC_SYNC)
      return (I_ERROR);
      }
   #endif
   if (! ctl->share) return (I_OK);

   /* first lock the file */
   if (! B_is_locked(ctl)) {      /* not locked */
      if (B_read_lock(ctl) == I_ERROR) {
         SET_CC(CC_SYNC)
         return (I_ERROR);
         }
      ctl->lock_state |= CTL_AUTO_READ_LOCK;
      }
   else if (ctl->current) return (I_OK);

   /* Read the control record directly from the disk */

   if (_cb_read(ctl->bfile->fd, 0L, sizeof(Ctldisk), &rec) != I_OK)
      ERR2X(EC_GETBLK, CC_SYNC)
   #if EXTRA_CHECKING
   if (B_calc_parity((unsigned short *)&rec, parity_words+1) != 0)
      ERR2X(EC_PARCHECK, CC_SYNC)
   #endif
   if (ctl->disk.update == rec.update) {
      ctl->current = 1;
      return (I_OK);
      }

   /*
      the btree file has been updated since the last time it was
      accessed by this program.  Save the new control record data
      and purge all the disk blocks associated with this file.
      */
   #if STRUCT_ASSIGN
   ctl->disk = rec;
   #else
   memcpy(ctl->disk,&rec,sizeof(Ctldisk));
   #endif

   if (purge_blks(ctl->bfile) == I_ERROR) ERR2X(EC_PURGEBLK, CC_SYNC)
   ctl->update = ctl->disk.update;
   ctl->current = 1;
   return (I_OK);
   } /* B_ctl_sync */
/**/
/*
        B_ctl_resync - synchronize disk file with memory

        This routine causes all data associated with this file to be flushed.
        This should be the last routine called before returning to the user's
        application.  If update is I_YES, the update field in the control
        record is incremented before it is flushed to the disk.  This routine
        is used only when changes have been made to the file.

        On entry, the file should be write locked.  If the file was auto locked
        it will be unlocked.

        Returns I_OK or I_ERROR.
*/


int B_ctl_resync(Ctlrec *ctl, int update, int delay_sync)
{
   int err;

   CB_ASSERT(B_is_write_locked(ctl));
   if (update) ctl->disk.update++;
   ctl->update = ctl->disk.update;
   if (! ctl->share) return (I_OK);
   err = I_OK;
   if (delay_sync && B_delay_sync && MANUAL_WRITE_LOCK(ctl) && ctl->current) {
      ctl->modified = 1;
      return (I_OK);
      }
   if (B_ctlsave(ctl) != I_OK) err = I_ERROR;
   if (flush_blks(ctl->bfile) == I_ERROR) {
      SET_EC(EC_FLUSHBLKS)
      SET_CC(CC_RESYNC)
      err = I_ERROR;
      }
   if (_cb_commitblk(ctl->bfile->fd) == I_ERROR) {
      SET_CC(CC_RESYNC)
      err = I_ERROR;
      }
   if (B_ctl_unlock(ctl) == I_ERROR) err = I_ERROR;
   ctl->modified = 0;
   return (err);
   } /* B_ctl_resync */

/**/
/*
        B_ctl_unlock - unlock the file

        This routine unlocks the control record if it was auto locked.
        Both read and write locks are removed.  Manual locks are not
        altered.
        Returns I_OK or I_ERROR.
*/


int B_ctl_unlock(Ctlrec *ctl)
{
   int err;
   int status = I_OK;
   int len;
   int fd;

   err = I_OK;
   fd = ctl->bfile->fd;

   ctl->current = 0;
   if ((ctl->lock_state & CTL_WLOCK) == CTL_AUTO_WRITE_LOCK) {
      if ((ctl->lock_state & CTL_RLOCK) != 0) {
         len = ctl->read_lock - sizeof(Ctldisk);
         if (len > 0)
            status = _cb_unlock_area(fd, (long) sizeof(Ctldisk), (long) len);
         else status = I_OK;
         len = ctl->disk.blksize - ctl->read_lock - 1;
         if (status != I_ERROR && len > 0)
            status = _cb_unlock_area(fd, ctl->read_lock+1, (long) len);
         }
      else
         status = _cb_unlock_area(fd, (long) sizeof(Ctldisk),
                                  (long)(ctl->disk.blksize - sizeof(Ctldisk)));
      if (status != I_ERROR) ctl->lock_state &= ~CTL_WLOCK;
      }
   if ((ctl->lock_state & CTL_RLOCK) == CTL_AUTO_READ_LOCK) {
      status = _cb_unlock_area(fd, (long)ctl->read_lock, 1L);
      if (status != I_ERROR) ctl->lock_state &= ~CTL_RLOCK;
      }
   if (status == I_ERROR) {
      SET_EC(EC_UNLOCK)
      SET_CC(CC_RESYNC)
      err = status;
      }
   if (err == I_OK && B_is_locked(ctl)) ctl->current = 1;
   return (err);
   } /* B_ctl_unlock */

/**/
/*
        B_ctl_unlock_all - unlock both automatic and manual locks

        Returns I_OK or I_ERROR.
*/


int B_ctl_unlock_all(Ctlrec *ctl)
{
   int err;
   int status = I_OK;
   int fd;
   int len;

   err = I_OK;
   fd = ctl->bfile->fd;

   ctl->current = 0;
   if ((ctl->lock_state & CTL_WLOCK) != 0) {
      if (ctl->modified) B_ctl_resync(ctl, I_NO, I_NO);
      if ((ctl->lock_state & CTL_RLOCK) != 0) {
         len = ctl->read_lock - sizeof(Ctldisk);
         if (len > 0)
            status = _cb_unlock_area(fd, (long) sizeof(Ctldisk), (long) len);
         else status = I_OK;
         len = ctl->disk.blksize - ctl->read_lock - 1;
         if (status != I_ERROR && len > 0)
            status = _cb_unlock_area(fd, ctl->read_lock+1, (long) len);
         }
      else
         status = _cb_unlock_area(fd, (long) sizeof(Ctldisk),
                                  (long)(ctl->disk.blksize - sizeof(Ctldisk)));
      if (status != I_ERROR) ctl->lock_state &= ~CTL_WLOCK;
      }
   if ((ctl->lock_state & CTL_RLOCK) != 0) {
      status = _cb_unlock_area(fd, ctl->read_lock, 1L);
      if (status != I_ERROR) ctl->lock_state &= ~CTL_RLOCK;
      }
   if (status == I_ERROR) {
      SET_EC(EC_UNLOCK)
      SET_CC(CC_RESYNC)
      err = status;
      }
   CB_ASSERT(ctl->modified == 0);
   return (err);
   } /* B_ctl_unlock_all */

/**/
/*
        B_read_lock - lock the file for reading

        Lock the file for read access by finding an unlocked byte in the
        control record and locking it.  Since write lock locks the entire
        control record, any byte will prevent write locking.  Searching
        begins at the byte used the last time the file was read locked.
        Each byte is tried and wrap arround occurs until we return to the
        starting location.  If a file has mutiple readers, saving the
        lock byte will tend to make each task use a different byte for
        locking and also allows tasks to come and go.  The maximum number
        of simultaneous readers is determined by the block size less the
        size of the control data.

        Returns I_OK or I_ERROR.
*/
int B_read_lock(Ctlrec *ctl)
{
   unsigned offset;
   int      status = I_OK;
   int      tries;
   int      fd;

   fd = ctl->bfile->fd;
   if (! ctl->share) return I_OK;
   if (ctl->lock_state != 0) return I_OK;
   tries = _cb_lock_tries;
   offset = ctl->read_lock;
   while (tries--) {
      do {
         status = _cb_lock_area(fd, (long)offset, 1L,
                1, _cb_lock_delay, _cb_lock_delay*2);
         if (status == I_OK) break;
         offset++;
         if (offset >= (unsigned) ctl->bfile->blksize) offset = sizeof(Ctldisk);
         } while (offset != ctl->read_lock);
      if (status == I_OK) break;
      _cb_delay(_cb_lock_delay);
      }
   if (status == I_ERROR) ERR2X(EC_LOCK, CC_LOCK)
   ctl->lock_state |= CTL_READ_LOCK;
   ctl->read_lock = offset;
   return (I_OK);
   }  /* B_read_lock */

/**/
/*
        B_write_lock - lock the file for writing

        Lock the file for write access by locking the entire control
        record.  If the file is already read locked, the control record
        is locked in two parts to skip the byte that was used for read
        locking.  This allows the library to work on systems that do not
        allow overlapping file locks.

        Returns I_OK or I_ERROR.
*/
int B_write_lock(Ctlrec *ctl)
{
   int      status;
   int      tries;
   int      fd;
   int      len1, len2;

   if (! ctl->share) return I_OK;
   if (B_is_write_lock(ctl)) return I_OK;
   fd = ctl->bfile->fd;
   tries = MULTI_LOCK_TRIES;
   if (B_is_read_lock(ctl)) {
      len1 = ctl->read_lock - sizeof(Ctldisk);
      len2 = ctl->disk.blksize - ctl->read_lock - 1;
      while (tries--) { /* must lock both at once */
         status = I_OK;
         if (len1 > 0)
            status = _cb_lock_area(fd, (long) sizeof(Ctldisk), (long)len1,
                _cb_lock_tries, _cb_lock_delay, _cb_lock_timeout);
         if (status == I_OK && len2 > 0) {
            status = _cb_lock_area(fd, (long)(ctl->read_lock+1), (long)len2,
               _cb_lock_tries, _cb_lock_delay, _cb_lock_timeout);
            if (status == I_ERROR && len1 > 0)
               _cb_unlock_area(fd, (long) sizeof(Ctldisk), (long) len1);
            }
         if (status == I_OK) break;
         }
      }
   else {
      status = _cb_lock_area(fd, (long)sizeof(Ctldisk),
               (long)(ctl->disk.blksize - sizeof(Ctldisk)),
               _cb_lock_tries, _cb_lock_delay, _cb_lock_timeout);
      }
   if (status == I_ERROR) ERR2X(EC_LOCK, CC_LOCK)
   ctl->lock_state |= CTL_WRITE_LOCK;
   return (I_OK);
   }  /* B_write_lock */

/**/
/*
        B_is_read_locked - return whether the file is safe to read

        Returns 1 or 0
*/

int B_is_read_locked(Ctlrec *ctl)
{
   if (ctl->share == 0 || ctl->lock_state != 0) return 1;
   return 0;
   }
/**/
/*
        B_is_write_locked - return whether the file is safe to write

        Returns 1 or 0
*/

int B_is_write_locked(Ctlrec *ctl)
{
   if (ctl->share == 0) return 1;
   if ((ctl->lock_state & CTL_WRITE_LOCK) != 0) return 1;
   return 0;
   }

/* determine whether the file is updated after every change
   when it is write locked */

I_EXPORT void I_ENTRY cb_set_flush(int delay_sync, int flush_blks)
{
   B_delay_sync = delay_sync;
   _b_no_flush = flush_blks;
   }
/* end ctlrec.c */
