/*      Copyright (c) 1987 Shrier and Deihl. Licensed to Mix Software, Inc.
        Copyright (c) 1995 Mix Software, Inc. Multi-user version

        cbmodify.c - CBT cbmodify and supporting functions
*/
#include "btree.h"
#include "cbtree.h"
/*
        Defined Functions

        cbmodify                modify current key and/or item
        locate_key              INTERNAL find key in leaf and get position
*/
/* find key in leaf, get position */
static int locate_key(Cbtree *tree, char *key, int keylen, Item item,
                      Blk_Nbr *block, int *index);
/**/
/*
        cbmodify - modify key and/or item to new key and item

        Returns OK or ERROR.
*/

I_EXPORT int I_ENTRY cbmodify(Cbtree *tree, char *oldkey, int oldkeylen,
             Item olditem, char *newkey, int newkeylen, Item newitem, int dup)
   /* tree      = tree to modify key in */
   /* oldkey    = existing key to modify */
   /* oldkeylen = existing key length */
   /* olditem   = existing item to modify */
   /* newkey    = new value of key */
   /* newkeylen = new key length */
   /* newitem   = new value of item */
   /* dup       = duplicate key allowed, I_YES or I_NO */
{
   Blk_Nbr old_block;      /* block number of old key */
   int old_index;          /* index of old key in block */
   Blk_Nbr new_block;      /* block number of new key */
   int new_index;          /* index of new key in block */
   Node *node;             /* node buffer */
   Key_Info key_info;      /* key information structure */
   Bfile *bfile;           /* buffered file pointer */
   Ctlrec *ctl;            /* control record */
   int status;

   ctl = tree->ctl;
   bfile = ctl->bfile;
   SET_FC(FC_CBMODIFY)
   if ((B_is_tree_valid(tree) == I_NO) || (B_is_ctl_valid(tree->ctl) == I_NO)
      || (B_is_key_valid(newkeylen) == I_NO)) return (I_ERROR);
   if (tree->readonly) ERR1X(EC_READONLY)

   /* write lock the tree for the duration of the operation */
   if (B_write_lock(ctl) != I_OK) return (I_ERROR);

   /* synchronize with the file and flush buffers if necessary */
   if (B_ctl_sync(ctl) == I_ERROR) {
      B_ctl_unlock(ctl);
      return (I_ERROR);
      }
   if (B_key_item_cmp(tree, oldkey, oldkeylen, olditem, newkey, newkeylen, newitem)
      == I_EQUAL) {    /* no change */
      status = B_inc_matches(oldkey, oldkeylen, olditem, tree);
      B_ctl_unlock(ctl);
      return status;
      }
   /*
           check for duplicate violations
   */
   if (locate_key(tree, newkey, newkeylen, newitem, &new_block, &new_index)
      != I_NOTFOUND) {
      SET_EC(EC_DUPKEYITEM)
      SET_CC(CC_INSNODE)
      B_ctl_unlock(ctl);
      return (I_ERROR);
      }
   key_info.key = newkey;
   key_info.keylen = (Keylen) newkeylen;
   key_info.item = newitem;
   key_info.dup = (short) dup;
   if ((node = (Node *) get_blk(bfile, new_block)) == (Node *) NULL) {
      SET_EC(EC_GETBLK)
      SET_CC(CC_FINDKEY)
      B_ctl_unlock(ctl);
      return (I_ERROR);
      }

   if ((*(tree->keycmp))(oldkey, oldkeylen, newkey, newkeylen) != I_EQUAL
      && B_is_dup_error(tree, &key_info, node, new_index) == I_YES) {
      rlse_blk(bfile, (char *) node);
      SET_EC(EC_DUPKEY)
      SET_CC(CC_INSNODE)
      B_ctl_unlock(ctl);
      return (I_ERROR);
      }

   /*
           if the new key goes into the same node as the old key, just
           remove the old key from the leaf.  Otherwise, do a full blown
           cbdelete.
   */
   if (locate_key(tree, oldkey, oldkeylen, olditem, &old_block, &old_index)
      != I_FOUND) {
      B_ctl_unlock(ctl);
      return (I_ERROR);
      }
   if (old_block == new_block) {
      if (B_inc_matches(oldkey, oldkeylen, olditem, tree) == I_ERROR) {
         SET_FC(FC_CBMODIFY)
         rlse_blk(bfile, (char *) node);
         B_ctl_resync(ctl, I_YES, I_YES); /* write buffers & ctl record */
         return (I_ERROR);
         }
      B_rmvkey_here(node, old_index);
      if (old_index < new_index) new_index--;
      }
   else {
      if (cbdelete(tree, oldkey, oldkeylen, olditem) != I_OK) {
         SET_FC(FC_CBMODIFY)
         rlse_blk(bfile, (char *) node);
         B_ctl_resync(ctl, I_YES, I_YES); /* write buffers & ctl record */
         return (I_ERROR);
         }
      SET_FC(FC_CBMODIFY)
      }

   /*
           if the new value of the modified key goes in the same leaf
           as the old value and there is enough room for the new value,
           just add the new value.  Otherwise, do a full blown cbinsert.
   */
   if ((old_block == new_block)
      && (B_key_fit(tree, &key_info, node, new_index, 0, node->key_cnt - 1)
      == I_YES)) {
      B_addkey_here(tree, &key_info, node, new_index);
      }
   else {
      if (cbinsert(tree, newkey, newkeylen, newitem, dup) != I_OK) {
         SET_FC(FC_CBMODIFY)
         rlse_blk(bfile, (char *) node);
         B_ctl_resync(ctl, I_YES, I_YES);
         return (I_ERROR);
         }
      SET_FC(FC_CBMODIFY)
      }
   if (old_block == new_block) put_blk(bfile, (char *) node, I_NO);
   else rlse_blk(bfile, (char *) node);
   B_ctl_resync(ctl, I_YES, I_YES); /* write buffers & ctl record to file */
   return (I_OK);
   } /* cbmodify */
/**/
/*
        locate_key - locate key in leaf block

        Locate_key finds the leaf block that contains the key and fills in
        the block number and index of the key.  If the key is not found,
        the I_NOTFOUND status is returned.

        Return I_FOUND, I_NOTFOUND or ERROR.
*/

static int locate_key(Cbtree *tree, char *key, int keylen, Item item,
                      Blk_Nbr *block, int *index)
   /* tree   = tree to locate key in */
   /* key    = key to locate */
   /* keylen = length of key */
   /* item   = item of key */
   /* block  = OUTPUT block containing key */
   /* index  = OUTPUT index of key in block */
{
   Blk_Nbr blk_nbr;        /* current block number */
   Node *node;             /* node buffer */
   int pos;                /* key position in node */
   int loc_stat;           /* location status */
   Bfile *bfile;           /* buffered file pointer */
   int   depth;            /* depth of tree */

   bfile = (tree->ctl)->bfile;
   blk_nbr = (tree->ctl)->disk.root;
   if (blk_nbr == (Blk_Nbr) 0) ERR2X(EC_EMPTYTREE, CC_FINDKEY)
   depth = tree->ctl->disk.depth;
   while (depth--) {
      if ((node = (Node *) get_blk(bfile, blk_nbr)) == (Node *) NULL)
         ERR2X(EC_GETBLK, CC_FINDKEY)
      loc_stat = B_loc_key(tree, key, keylen, item, node, &pos);
      if (node->low_path == LEAF_NODE) {
         rlse_blk(bfile, (char *) node);
         *block = blk_nbr;
         *index = pos;
         if (loc_stat == I_EQUAL) return (I_FOUND);
         else {
            SET_EC(EC_NOKEY)
            SET_CC(CC_FINDKEY)
            return (I_NOTFOUND);
            }
         }
      if (loc_stat == I_EQUAL) blk_nbr = Ikey_Child(node, pos);
      else {
         if (pos == 0) blk_nbr = node->low_path;
         else blk_nbr = Ikey_Child (node, (pos - 1));
         }
      rlse_blk(bfile, (char *) node);
      }
   return (I_ERROR);
   } /* locate_key */

/* end cbmodify.c */
