/*      Copyright (c) 1987 Shrier and Deihl. Licensed to Mix Software, Inc.
        Copyright (c) 1995 Mix Software, Inc. Multi-user version

        cbdelete.c - CBT cbdelete function
*/
#include "btree.h"
/*
        Defined Functions (INTERNAL functions for use by CBT only)

        cbdelete                delete a key/item pair
        B_del_tree              (INTERNAL) delete key from tree
        del_subtree             (INTERNAL) delete key from subtree
        rmv_unused_keys         (INTERNAL) remove keys of deleted children
        loc_unused_key          (INTERNAL) locate first unused key in node
        loc_next_child          (INTERNAL) locate child after removed child
*/
extern Cbtree *B_valid_trees;

/* delete key from subtree */
static int del_subtree(Cbtree *tree, Key_Info *key_info, Blk_Nbr blk_nbr,
                       Blk_Nbr parent, int par_pos);

/* remove keys of deleted children */
static int rmv_unused_keys(Cbtree *tree, Blk_Nbr blk_nbr, int pos,
                           Blk_Nbr parent, int par_pos);

/* locate first unused key in node */
static int loc_unused_key(Cbtree *tree, Node *node, int pos, int *unused_pos);

/* locate child after removed child */
static int loc_next_child(Cbtree *tree, Node *node, int unused_pos,
                          Blk_Nbr *next_child);
/**/
/*
        cbdelete - delete a key/item pair

        Cbdelete removes the key/item pair that EXACTLY matches the
        key/item pair passed in.  If the key/item is not in the tree,
        it is an error.  As the nodes of the tree become empty, they
        are joined in such a fashion to maintain a balanced tree.

        If any file cursor or mark is pointing to the deleted pair,
        the file cursor or mark is advanced to the next pair (or I_EOI).
        Otherwise, no file cursors or marks are moved.

        Returns OK or ERROR.
*/

I_EXPORT int I_ENTRY cbdelete(Cbtree *tree, char *key, int keylen, Item item)
   /* tree   = tree to delete key from */
   /* key    = key to be deleted */
   /* keylen = length of key */
   /* item   = item to be deleted */
{
   int status;             /* subroutine return status */
   Key_Info key_info;      /* key to be deleted */
   Ctlrec *ctl;            /* control record */

   SET_FC(FC_CBDELETE)
   if ((B_is_tree_valid(tree) == I_NO) || (B_is_ctl_valid(tree->ctl) == I_NO))
      return (I_ERROR);
   if (tree->readonly) ERR1X(EC_READONLY)

   key_info.key = key;
   key_info.keylen = (Keylen) keylen;
   key_info.item = item;
   ctl = tree->ctl;

   /* write lock the tree for the duration of the operation */
   if (B_write_lock(ctl) != I_OK) return (I_ERROR);

   /* synchronize with the file and flush buffers if necessary */
   if (B_ctl_sync(ctl) == I_ERROR) {
      B_ctl_unlock(ctl);
      return (I_ERROR);
      }
   /* update file cursors and marks */
   status = B_inc_matches(key, keylen, item, tree);
   /*
                there are 2 cases for delete, deleting from a tree that
                already contains some key/item pairs or deleting from
                an empty tree.
        */
   if ((tree->ctl)->disk.root != (Blk_Nbr) 0) {
      if (B_del_tree(tree, &key_info) == I_ERROR) status = I_ERROR;
      }
   else {
      SET_EC(EC_EMPTYTREE)
      SET_CC(CC_DELEMPTY)
      status = I_ERROR;
      }

   ctl->disk.keycount--;
   B_ctl_resync(ctl, I_YES, I_YES); /* write buffers & ctl record to file */
   return (status);
   } /* cbdelete */
/**/
/*
        B_del_tree - delete key/item pair from non-empty btree

        Returns OK or ERROR.
*/

int B_del_tree(Cbtree *tree, Key_Info *key_info)
   /* tree     = tree to delete key from */
   /* key_info = key to delete */
{
   Ctlrec *ctl;            /* control structure for tree */
   Blk_Nbr low_path;       /* low path of root node */
   int status;             /* subroutine return status */

   ctl = tree->ctl;
   if (B_get_low_path(tree, ctl->disk.root, &low_path) == I_ERROR)
      return (I_ERROR);
   if (low_path == LEAF_NODE) {
      if ((status = B_del_node(tree, key_info, ctl->disk.root, (Blk_Nbr) 0, 0))
         == I_ERROR) return (I_ERROR);
      }
   else if (low_path == UNUSED_NODE) ERR2X(EC_CORRUPT, CC_DELUNUSED)
   else {
      if ((status = del_subtree(tree, key_info, ctl->disk.root, (Blk_Nbr) 0, 0))
         == I_ERROR) return (I_ERROR);
      }
   if (status == JOIN) {
      if ((status = B_rmv_root(tree)) == I_ERROR) return (I_ERROR);
      }
   return (status);
   } /* B_del_tree */
/**/
/*
        del_subtree - delete key from a subtree

        del_subtree handles deleting a key from a subtree and handles all
        contengencies.  If the root of the subtree was joined with one or
        more of its siblings, the number of node joins is returned in
        join_cnt.

        Returns OK, JOIN, or ERROR.
*/

static int del_subtree(Cbtree *tree, Key_Info *key_info, Blk_Nbr blk_nbr,
                       Blk_Nbr parent, int par_pos)
   /*ree *tree       = tree containing subtree */
   /*_Info *key_info = key to delete */
   /*_Nbr blk_nbr    = block number of target subtree */
   /*_Nbr parent     = block number of parent node */
   /* par_pos        = position of target in parent */
{
   int status;             /* insertion status */
   int pos;                /* key position index */
   Blk_Nbr child;          /* block number of child */
   Blk_Nbr low_path;       /* low path in child */

   if ((child = B_loc_subtree(tree, key_info, blk_nbr, &pos))
      == (Blk_Nbr) I_ERROR) return (I_ERROR);
   if (B_get_low_path(tree, child, &low_path) == I_ERROR) return (I_ERROR);
   if (low_path == LEAF_NODE) {
      if ((status = B_del_node(tree, key_info, child, blk_nbr, pos))
         == I_ERROR) return (I_ERROR);
      }
   else if (low_path == UNUSED_NODE) ERR2X(EC_CORRUPT, CC_DELNODE)
   else {
      if ((status = del_subtree(tree, key_info, child, blk_nbr, pos))
         == I_ERROR) return (I_ERROR);
      }
   if (status == JOIN) {
      status = rmv_unused_keys(tree, blk_nbr, pos, parent, par_pos);
      }
   return (status);
   } /* del_subtree */
/**/
/*
        rmv_unused_keys - remove keys of deleted children

        After one or more nodes have joined into one node, rmv_unused_keys
        takes the references to the removed nodes from the interior node
        that used to be their parent node.

        Returns JOIN, OK, or ERROR.
*/

static int rmv_unused_keys(Cbtree *tree, Blk_Nbr blk_nbr, int pos,
                           Blk_Nbr parent, int par_pos)
   /* tree    = tree containing nodes to remove */
   /* blk_nbr = block to remove keys from */
   /* pos     = position of joined child */
   /* parent  = parent of blk_nbr */
   /* par_pos = position in parent of blk_nbr */
{
   Node *node;             /* node buffer */
   int unused_pos;         /* position of unused key */
   Blk_Nbr child;          /* child being removed */
   Blk_Nbr next_child;     /* next valid child after unused */
   int status;             /* subroutine return status */
   Bfile *bfile;           /* buffered file pointer */

   bfile = (tree->ctl)->bfile;
   if ((node = (Node *) get_blk(bfile, blk_nbr)) == (Node *) NULL)
      ERR2X(EC_GETBLK, CC_RMVUNUSEDKEYS)
   if (loc_unused_key(tree, node, pos, &unused_pos) == I_ERROR) {
      rlse_blk(bfile, (char *) node);
      return (I_ERROR);
      }
   CB_ASSERT(unused_pos < node->key_cnt);
   if (loc_next_child(tree, node, unused_pos, &next_child) == I_ERROR) {
      rlse_blk(bfile, (char *) node);
      return (I_ERROR);
      }
   if (unused_pos == -1) child = node->low_path;
   else child = Ikey_Child(node, unused_pos);
   while (child != next_child) {
      CB_ASSERT(child != (Blk_Nbr) 0);
      if (unused_pos >= node->key_cnt) break;
      B_rmvkey_here(node, unused_pos);
      if (unused_pos == -1) child = node->low_path;
      else child = Ikey_Child(node, unused_pos);
      }
   status = I_OK;
   if (parent == (Blk_Nbr) 0) {
      if (node->key_cnt == 0) status = JOIN;
      }
   else {
      if (node->key_cnt == -1) {
         if (B_release_node(tree, blk_nbr, node) == I_ERROR) {
            rlse_blk(bfile, (char *) node);
            return (I_ERROR);
            }
         status = JOIN;
         }
      }
   if (put_blk(bfile, (char *) node, I_NO) == I_ERROR)
      ERR2X(EC_PUTBLK, CC_RMVUNUSEDKEYS)
   if ((parent != (Blk_Nbr) 0) && (status == I_OK))
      return (B_compress_tree(tree, parent, par_pos));
   else return (status);
   } /* rmv_unused_keys */
/**/
/*
        loc_unused_key - locate first unused key in interior node

        Loc_unused_key locates the first unused key in an interior node
        by finding the first key that points to an unused node.  The only
        two keys that can be the first key are the one at pos or pos + 1.
        The position of the first unused key is returned in unused_pos.

        Returns OK or ERROR.
*/

static int loc_unused_key(Cbtree *tree, Node *node, int pos, int *unused_pos)
   /* tree       = tree containing interior node */
   /* node       = interior node buffer */
   /* pos        = position of subtree in node */
   /* unused_pos = position of first unused key */
{
   Blk_Nbr child;                  /* child block number */
   Blk_Nbr low_path;               /* low path in child */

   *unused_pos = pos - 1;
   if (*unused_pos < -1) *unused_pos = -1;
   if (*unused_pos == -1) child = node->low_path;
   else child = Ikey_Child(node, *unused_pos);
   if (B_get_low_path(tree, child, &low_path) == I_ERROR) return (I_ERROR);
   if (low_path == UNUSED_NODE) return (I_OK);
   (*unused_pos)++;
   child = Ikey_Child(node, *unused_pos);
   if (B_get_low_path(tree, child, &low_path) == I_ERROR) return (I_ERROR);
   if (low_path == UNUSED_NODE) return (I_OK);
   (*unused_pos)++;
   return (I_OK);
   } /* loc_unused_key */
/**/
/*
        loc_next_child - locate next child after removed children

        Loc_next_child returns the block number of the first non-removed
        child node after a removed child node.  The block number is
        returned in next_child.

        Returns OK or ERROR.
*/

static int loc_next_child(Cbtree *tree, Node *node, int unused_pos,
                          Blk_Nbr *next_child)
   /* tree       = tree containing interior node */
   /* node       = interior node buffer */
   /* unused_pos = position of first unused key */
   /* next_child = first non_removed child */
{
   Blk_Nbr child;                  /* child block number */

   if (unused_pos == -1) {
      if (node->key_cnt == 0) {
         if (B_get_low_path(tree, node->right, next_child) == I_ERROR)
            return (I_ERROR);
         }
      else *next_child = Ikey_Child(node, 0);
      }
   else {
      if (unused_pos == 0) child = node->low_path;
      else child = Ikey_Child(node, unused_pos - 1);
      if (B_get_right(tree, child, next_child) == I_ERROR) return (I_ERROR);
      }
   return (I_OK);
   } /* loc_next_child */

/**/
/*
        B_inc_matches - move all matching cursors and marks to the next key

        Seaches all open trees for cursors and marks that match the item
        that was modified or deleted.  Each match is moved to the next
        key-item pair.

        Returns OK or ERROR.
*/

int  B_inc_matches(char *key, int keylen, Item item, Cbtree *tree)
{
   Cbtree *t;
   t = B_valid_trees;
   while (t != NULL) {
      if (t == tree && t->cur_key.item == item &&
          (*(t->keycmp))(key, keylen, t->cur_key.key, t->cur_key.keylen)
          == I_EQUAL) {
         if (B_key_sync(t, &t->cur_key) == I_ERROR) return I_ERROR;
         if (B_pos_nextkey(t) == I_ERROR) return I_ERROR;
         }
      if (t == tree && t->mrk_key.item == item &&
          (*(t->keycmp))(key, keylen, t->mrk_key.key, t->mrk_key.keylen)
          == I_EQUAL) {
         if (B_key_sync(t, &t->mrk_key) == I_ERROR) return I_ERROR;
         if (B_pos_nextmark(t) == I_ERROR) return I_ERROR;
         }
      t = t->next;
      }
   return I_OK;
   }

/* end cbdelete.c */
