/*      Copyright (c) 1994 Mix Software, Inc.

        btree.h - include file for btree.c module
*/

#ifndef _BTREE_H
#define _BTREE_H
#define CDT_SRC     1
#if !defined(CDTDEF_H_INCLUDED)
#include "cdtdef.h"
#endif

#ifndef _BUFPOOL_H
#include "bufpool.h"
#endif
#ifndef _CTLREC_H
#include "ctlrec.h"
#endif

#if !defined(_ERROR_VARS)
#include "cberr.h"
#endif
#if defined(cbconcode)
#undef cbconcode
#undef cbfcncode
#undef cberrcode
I_EXPORT_DATA int cbconcode;  /* context code - what it was doing */
I_EXPORT_DATA int cbfcncode;  /* function code - who detected the error */
I_EXPORT_DATA int cberrcode;  /* error code - the error itself */
#endif

/* do extra runtime checking on internal function calls */
#define EXTRA_CHECKING 0
/* compiler support for structure assignment */
#define STRUCT_ASSIGN  1

/* error code macros */
#if defined(WIN16_DLL)
   #define ERR1(code) *(_cb_inst->errcode) = code;
   #define ERR2(code, cc) {*(_cb_inst->errcode) = code; *(_cb_inst->concode) = cc;}
   #define ERR1X(code) {*(_cb_inst->cberrcode) = code; return I_ERROR;}
   #define ERR2X(code, cc) {*(_cb_inst->errcode) = code; *(_cb_inst->concode) = cc; return I_ERROR;}
   #define SET_FC(code) *(_cb_inst->fcncode) = code;
   #define SET_CC(code) *(_cb_inst->concode) = code;
   #define SET_EC(code) *(_cb_inst->errcode) = code;
   #define GET_FC       *(_cb_inst->fcncode)
   #define GET_CC       *(_cb_inst->concode)
   #define GET_EC       *(_cb_inst->errcode)
#else
   #define ERR1(code) cberrcode = code;
   #define ERR2(code, cc) {cberrcode = code; cbconcode = cc;}
   #define ERR1X(code) {cberrcode = code; return I_ERROR;}
   #define ERR2X(code, cc) {cberrcode = code; cbconcode = cc; return I_ERROR;}
   #define SET_FC(code) cbfcncode = code;
   #define SET_CC(code) cbconcode = code;
   #define SET_EC(code) cberrcode = code;
   #define GET_FC       cbfcncode
   #define GET_CC       cbconcode
   #define GET_EC       cberrcode
#endif

#if defined(DEBUG)
   #undef  EXTRA_CHECKING
   #define EXTRA_CHECKING 1
   #define CB_ASSERT(exp) if (!(exp)) B_assert_msg(__FILE__, __LINE__)
   #define CB_CHECK1(exp, con) if (exp) {cbconcode = con; return(I_ERROR);}
   #define CB_CHECK2(exp, err, con) if (exp) {cbconcode = con; cberrcode = err; return(I_ERROR);}
   extern void B_assert_msg(char *file, int line);
#else
   #define CB_ASSERT(exp)
   #if EXTRA_CHECKING
      #define CB_CHECK1(exp, con) if (exp) {cbconcode = con; return(I_ERROR);}
      #define CB_CHECK2(exp, err, con) if (exp) {cbconcode = con; cberrcode = err; return(I_ERROR);}
   #else
      #define CB_CHECK1(exp, con)
      #define CB_CHECK2(exp, err, con)
   #endif
#endif

/*
        The internal structure of a Cbtree object.

        When curblock == NONE, the curindex is meaningless.
        When mrkblock == NONE, the mrkindex is meaningless.
*/

typedef short Offset;         /* byte offset from some beginning */
typedef short Keylen;         /* length of a key */
typedef short Pair_Nbr;       /* nbr of I_Pair or L_Pair within node */
typedef unsigned long Item;   /* type for a key's associated item */

#define NONE    (-1)          /* no value */

/*
        Structure of a Node
*/
#define LEAF_NODE       ((Blk_Nbr) -1)  /* this node is a leaf node */
#define UNUSED_NODE     ((Blk_Nbr) 0)   /* this node is vacant */

typedef struct i_pairs {      /* pairs in interior nodes */
   Offset key_offset;         /* offset to key in heap */
   Keylen key_len;            /* nbr of bytes in key */
   Item key_item;             /* Item value associated with this key */
   Blk_Nbr child;             /* block nbr of child of this key */
   } I_Pairs;

typedef struct l_pairs {      /* pairs in leaf nodes */
   Offset key_offset;         /* offset to key in heap */
   Keylen key_len;            /* nbr of bytes in key */
   Item key_item;             /* Item value associated with this key */
   } L_Pairs;

/*
        define a constant for the i_node and l_node array size.  This
        size has nothing to do with the actual size of the array but it
        gets around a bug (feature) in the turbo and microsoft C compiler.
*/
#define XX      1

typedef struct {
   Blk_Nbr low_path;       /* low path, or LEAF_NODE, or UNUSED_NODE */
   Blk_Nbr left;           /* blk number of left sibling or next unused node */
   Blk_Nbr right;          /* block number of right sibling */
   Pair_Nbr key_cnt;       /* nbr of keys currently used in node */
   Offset heap_end;        /* heap end's offset from beginning of node */
   union {
      I_Pairs i_node[XX];     /* interior node key/block pairs */
      L_Pairs l_node[XX];     /* leaf node key/item pairs */
      } pairs;
   } Node;
#define NODEHDRSIZE ((3*sizeof(Blk_Nbr)) + sizeof(Pair_Nbr) + sizeof(Offset))
#define Join_Limit(c)      ((((c)->disk.blksize - NODEHDRSIZE) * 3) / 4)
#define Lkey_Addr(n, p)    ((char *)(n) + (n)->pairs.l_node[p].key_offset)
#define Lkey_Offset(n, p)  ((n)->pairs.l_node[p].key_offset)
#define Lkey_Len(n, p)     ((n)->pairs.l_node[p].key_len)
#define Lkey_Item(n, p)    ((n)->pairs.l_node[p].key_item)
#define Ikey_Addr(n, p)    ((char *)(n) + (n)->pairs.i_node[p].key_offset)
#define Ikey_Offset(n, p)  ((n)->pairs.i_node[p].key_offset)
#define Ikey_Len(n, p)     ((n)->pairs.i_node[p].key_len)
#define Ikey_Item(n, p)    ((n)->pairs.i_node[p].key_item)
#define Ikey_Child(n, p)   ((n)->pairs.i_node[p].child)

/*      define block result codes */
#define SPLIT           3       /* node split */
#define JOIN            4       /* node join */
#define SHIFT           5       /* keys shifted */
#define SHIFT_INS       6       /* keys shifted, key_info inserted */

/*      collect up all the constant information about a key being inserted
        into the btree.
*/

typedef struct {
   char     *key;
   Keylen   keylen;
   Item     item;
   short    dup;
   Blk_Nbr  child;
   } Key_Info;

/*      The internal structure of a Cbtree object.

        When curblock == NONE, the curindex is meaningless.
        When mrkblock == NONE, the mrkindex is meaningless.
*/

typedef struct {
   Blk_Nbr  block;         /* block number */
   int      status;        /* status (BOI, OK, or EOI) */
   int      index;         /* file cursor - index of l_pair */
   char     *key;          /* current key */
   Keylen   keylen;        /* length of key */
   Item     item;          /* item for key */
   Keylen   maxlen;        /* length of key buffer */
   unsigned long update;   /* file update count */
   } Saved_Key;

typedef struct cbtree {
   struct cbtree *next;    /* next in list of valid trees */
   Ctlrec *ctl;            /* pointer to control record */
   Cmp_Fn_Ptr keycmp;      /* key compare */
   Saved_Key cur_key;      /* key/item of current key */
   Saved_Key mrk_key;      /* key/item of marked key */
   int readonly;           /* opened for read only access */
   } Cbtree;

/*      internal global variables */

extern Cbtree *B_valid_trees;    /* list of valid trees */
extern Pool_Obj B_cbtpool;       /* the buffer pool */
extern int _cb_lock_tries;       /* number of locking attempts */
extern long _cb_lock_delay;      /* delay between tries */
extern long _cb_lock_timeout;    /* total timeout */

/*      tracing flag variable and flag names */

extern int cbttrace;       /* tracing flags */
#define TF_CALL         0x0001  /* trace subroutine calls */
#define TF_RETURN       0x0002  /* trace subroutine returns */

/*      maximum number of attempts to lock a file */
#define MAX_LOCK_TRIES  3
#define LOCK_DELAY      1
/*      minimum space in control record for locks */
#define MIN_LOCKS       16

/*      internal subroutine prototypes */

int  B_addkey_here(Cbtree *tree, Key_Info *key_info, Node *node, int pos);
int  B_add_low_key(Cbtree *tree, Blk_Nbr blk_nbr, Node *node, Blk_Nbr child,
       Blk_Nbr parent, int par_pos, int *pos);
int  B_calc_free_space(Cbtree *tree, Node *node);
int  B_calc_key_space(Node *node, int start, int end);
int  B_calc_used_space(Cbtree *tree, Node *node);
void B_clearbufs(void);
int  B_cmp_low_path(Cbtree *tree, Key_Info *key_info, Node *node, int *status);
int  B_combined_space(Cbtree *tree, Blk_Nbr left, Blk_Nbr right);
int  B_compress_tree(Cbtree *tree, Blk_Nbr parent, int par_pos);
int  B_cur_key_match(Cbtree *tree, char *key, int keylen);
int  B_del_node(Cbtree *tree, Key_Info *key_info, Blk_Nbr blk_nbr,
       Blk_Nbr parent, int par_pos);
int  B_del_tree(Cbtree *tree, Key_Info *key_info);
int  B_enter(Cbtree *tree);
int  B_exit(Cbtree *tree);
int  B_fix_left(Cbtree *tree, Blk_Nbr blk_nbr, Blk_Nbr left);
int  B_fix_right(Cbtree *tree, Blk_Nbr blk_nbr, Blk_Nbr right);
int  B_flush(Cbtree *tree);
int  B_get_curitem(Cbtree *tree, Item *item_ptr);
char *B_get_empty_block(Cbtree *tree, Blk_Nbr *block_number);
int  B_get_key_cnt(Cbtree *tree, Blk_Nbr blk_nbr, int *key_cnt);
int  B_get_left(Cbtree *tree, Blk_Nbr blk_nbr, Blk_Nbr *left);
int  B_get_low_path(Cbtree *tree, Blk_Nbr blk_nbr, Blk_Nbr *low_path);
int  B_get_right(Cbtree *tree, Blk_Nbr blk_nbr, Blk_Nbr *right);
int  B_inc_matches(char *key, int keylen, Item item, Cbtree *tree);
void B_init_leaf(Node *buf, Blk_Nbr left, Blk_Nbr right, int node_size);
void B_init_node(Node *buf, Blk_Nbr low_path, Blk_Nbr left, Blk_Nbr right,
       int node_size);
int  B_ins_complications(Cbtree *tree, Key_Info *key_info, Node *node, int pos);
int  B_ins_new_tree(Cbtree *tree, Key_Info *key_info);
int  B_ins_node(Cbtree *tree, Key_Info *key_info, Blk_Nbr blk_nbr,
       Blk_Nbr parent, int par_pos);
int  B_ins_tree(Cbtree *tree, Key_Info *key_info);
int  B_is_dup_error(Cbtree *tree, Key_Info *key_info, Node *node, int pos);
int  B_is_k_info_dup(Cbtree *tree, Key_Info *key_info, Node *node, int pos,
       int *offset);
int  B_is_key_valid(int keyln);
int  B_is_tree_valid(Cbtree *tree);
int  B_join_node(Cbtree *tree, Blk_Nbr right);
int  B_key_fit(Cbtree *tree, Key_Info *key_info, Node *node, int pos, int start,
       int end);
int  B_key_info_loc(Cbtree *tree, Key_Info *key_info, Node *node, int *pos);
int  B_key_item_cmp(Cbtree *tree, char *keya, int lena, Blk_Nbr itema, char *keyb,
       int lenb, Blk_Nbr itemb);
int  B_key_match(Cbtree *tree, Node *nodea, int posa, Node *nodeb, int posb);
int  B_key_save(Node *node, int pos, Saved_Key *keyptr);
int  B_key_sync(Cbtree *tree, Saved_Key *keyptr);
int  B_left_offload(Cbtree *tree, Key_Info *key_info, Node *node, int pos,
       Blk_Nbr parent, int par_pos);
int  B_loc_key(Cbtree *tree, char *key, int keylen, Item item, Node *node,
       int *pos);
int  B_loc_leaf_find(Cbtree *tree, char *key, int keylen, Item item,
       Saved_Key *keyptr, int copy);
Blk_Nbr B_loc_subtree(Cbtree *tree, Key_Info *key_info, Blk_Nbr blk_nbr,
       int *pos);
int  B_loc_low_blk(Cbtree *tree, Blk_Nbr blk_nbr, Blk_Nbr *low_path_blk);
int  B_make_key_fit(Cbtree *tree, Key_Info *key_info, Blk_Nbr blk_nbr,
       Node *node, int pos, Blk_Nbr parent, int par_pos);
int  B_make_root(Cbtree *tree);
int  B_move_left(Cbtree *tree, Blk_Nbr blk_nbr, int key_count);
int  B_move_right(Cbtree *tree, Blk_Nbr blk_nbr, int key_count);
int  B_node_keycmp(Cbtree *tree, Key_Info *key_info, Node *node, int pos);
int  B_pos_firstkey(Cbtree *tree);
int  B_pos_lastkey(Cbtree *tree);
int  B_pos_nextkey(Cbtree *tree);
int  B_pos_nextmark(Cbtree *tree);
int  B_pos_prevkey(Cbtree *tree);
int  B_release_node(Cbtree *tree, Blk_Nbr blk_nbr, Node *node);
int  B_right_offload(Cbtree *tree, Key_Info *key_info, Node *node, int pos,
       Blk_Nbr parent, int par_pos);
int  B_rmv_root(Cbtree *tree);
void B_rmvkey_here(Node *node, int pos);
void B_setbuf_size(int bufcnt, int bufsize);
Cbtree *B_treealloc(Cmp_Fn_Ptr keycmp, Ctlrec *ctl);
int  B_treefree(Cbtree *tree);
#endif  /* _BTREE_H */

/* end btree.h */
